% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelHD.R
\name{modelHD}
\alias{modelHD}
\title{Fitting height-diameter models}
\usage{
modelHD(D, H, method = NULL, useWeight = FALSE, drawGraph = FALSE, plot = NULL)
}
\arguments{
\item{D}{Vector with diameter measurements (in cm). NA values are accepted but a
minimum of 10 valid entries (i.e. having a corresponding height in H) is required.}

\item{H}{Vector with total height measurements (in m). NA values are accepted but a minimum of 10 valid entries
(i.e. having a corresponding diameter in D) is required.}

\item{method}{Method used to fit the relationship.
To be chosen between:
\itemize{
\item log1, log2
\itemize{
\item log 1: \eqn{(log(H) = a+ b*log(D))} (equivalent to a power model)
\item log 2: \eqn{(log(H) = a+ b*log(D) + c*log(D)^2)}
}
\item weibull: \eqn{H = a*(1-exp(-(D/b)^c))}
\item michaelis: \eqn{H = (A * D)/(B + D)}
}

If \code{NULL}, all the methods will be compared.}

\item{useWeight}{If weight is \code{TRUE}, model weights will be \eqn{(D^2)*H} (i.e. weights are proportional to tree
volume, so that larger trees have a stronger influence during the construction of the model).}

\item{drawGraph}{If \code{TRUE}, a graphic will illustrate the relationship between H and D. Only if argument \code{plot} is null.}

\item{plot}{(optional) Plot ID, must be either one value, or a vector of the same length as D. This argument is used to build
stand-specific HD models.}
}
\value{
If plot is NULL or has a single value, a single list is returned. If there is more than one plot,
multiple embedded lists are returned with plots as the list names.
Returns a list if the parameter model is not null:
\itemize{
\item \code{input}: list of the data used to construct the model (list(H, D))
\item \code{model}: outputs of the model (same outputs as given by \code{\link[stats:lm]{stats::lm()}}, \code{\link[stats:nls]{stats::nls()}})
\item \code{RSE}: Residual Standard Error of the model
\item \code{RSElog}: Residual Standard Error of the log model (\code{NULL} if other model)
\item \code{residuals}: Residuals of the model
\item \code{coefficients}: Coefficients of the model
\item \code{R.squared}: \eqn{R^2} of the model
\item \code{formula}: Formula of the model
\item \code{method}: Name of the method used to construct the model
\item \code{predicted}: Predicted height values
}

If the parameter model is null, the function return a graph with all the methods for
comparison, the function also returns a data.frame with:
\itemize{
\item \code{method}: The method that had been used to construct the graph
\item \code{color}: The color of the curve in the graph
\item \code{RSE}: Residual Standard Error of the model
\item \code{RSElog}: Residual Standard Error of the log model (\code{NULL} if other model)
\item \code{Average_bias}: The average bias for the model
}
}
\description{
This function fits and compares (optional) height-diameter models.
}
\details{
All the back transformations for log-log models are done using the Baskerville correction (\eqn{0.5 * RSE^2},
where RSE is the Residual Standard Error).
}
\examples{

# Load a data set
data(NouraguesHD)

# To model the height from a dataset
\donttest{
HDmodel <- modelHD(D = NouraguesHD$D, H = NouraguesHD$H, drawGraph = TRUE)
}

# If the method needed is known
HDmodel <- modelHD(D = NouraguesHD$D, H = NouraguesHD$H, method = "weibull", drawGraph = TRUE)
HDmodel <- modelHD(D = NouraguesHD$D, H = NouraguesHD$H, method = "log1", drawGraph = TRUE)

# Using weights
HDmodel <- modelHD(
  D = NouraguesHD$D, H = NouraguesHD$H, method = "weibull", useWeight = TRUE,
  drawGraph = TRUE
)
}
\seealso{
\code{\link[=retrieveH]{retrieveH()}}
}
\author{
Maxime REJOU-MECHAIN, Arthur PERE, Ariane TANGUY
}
