% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sampleLCA.R
\name{sampleLCA}
\alias{sampleLCA}
\title{Telescoping sampling of the LCA model where a prior on the number
of components K is specified.}
\usage{
sampleLCA(
  y,
  S,
  pi,
  eta,
  a0,
  M,
  burnin,
  thin,
  Kmax,
  G = c("MixDynamic", "MixStatic"),
  priorOnK,
  priorOnWeights,
  verbose = FALSE
)
}
\arguments{
\item{y}{A numeric matrix; containing the data.}

\item{S}{A numeric matrix; containing the initial cluster
assignments.}

\item{pi}{A numeric vector; containing the initial cluster-specific
success probabilities.}

\item{eta}{A numeric vector; containing the initial cluster sizes.}

\item{a0}{A numeric vector; containing the parameters of the prior on the
cluster-specific success probabilities.}

\item{M}{A numeric scalar; specifying the number of recorded
iterations.}

\item{burnin}{A numeric scalar; specifying the number of burn-in
iterations.}

\item{thin}{A numeric scalar; specifying the thinning used for the
iterations.}

\item{Kmax}{A numeric scalar; the maximum number of components.}

\item{G}{A character string; either \code{"MixDynamic"} or \code{"MixStatic"}.}

\item{priorOnK}{A named list; providing the prior on the number of components K, see \code{\link[=priorOnK_spec]{priorOnK_spec()}}.}

\item{priorOnWeights}{A named list; providing the prior on the mixture weights.}

\item{verbose}{A logical; indicating if some intermediate clustering
results should be printed.}
}
\value{
A named list containing:
\itemize{
\item \code{"Pi"}: sampled component-specific success probabilities.
\item \code{"Eta"}: sampled weights.
\item \code{"S"}: sampled assignments.
\item \code{"Nk"}: number of observations assigned to the different components, for each iteration.
\item \code{"K"}: sampled number of components.
\item \code{"Kplus"}: number of filled, i.e., non-empty components, for each iteration.
\item \code{"e0"}: sampled Dirichlet parameter of the prior on the weights (if \eqn{e_0} is random).
\item \code{"alpha"}: sampled Dirichlet parameter of the prior on the weights (if \eqn{\alpha} is random).
\item \code{"acc"}: logical vector indicating acceptance in the Metropolis-Hastings step when sampling either \eqn{e_0} or \eqn{\alpha}.
}
}
\description{
\itemize{
\item The MCMC scheme is implemented as suggested in Frühwirth-Schnatter et al (2021).
\item The priors on the model parameters are specified as in Frühwirth-Schnatter et al (2021),
see the vignette for details and notation.
}
}
\examples{
if (requireNamespace("poLCA", quietly = TRUE)) {
    data("carcinoma", package = "poLCA")
    y <- carcinoma
    N <- nrow(y)
    r <- ncol(y)
    
    M <- 200
    thin <- 1
    burnin <- 100
    Kmax <- 50  
    Kinit <- 10
    
    G <- "MixDynamic"
    priorOnAlpha <- priorOnAlpha_spec("gam_1_2")
    priorOnK <- priorOnK_spec("Pois_1")
    
    cat <- apply(y, 2, max)
    a0 <- rep(1, sum(cat))

    cl_y <- kmeans(y, centers = Kinit, iter.max = 20)
    S_0 <- cl_y$cluster
    eta_0 <- cl_y$size/N

    pi_0 <- do.call("cbind", lapply(1:r, function(j) {
        prop.table(table(S_0, y[, j]), 1)
    }))

    result <- sampleLCA(
        y, S_0, pi_0, eta_0, a0, 
        M, burnin, thin, Kmax, 
        G, priorOnK, priorOnAlpha)

    K <- result$K
    Kplus <- result$Kplus   
    
    plot(K, type = "l", ylim = c(0, max(K)),  
         xlab = "iteration", main = "",
         ylab = expression("K" ~ "/" ~ K["+"]), col = 1)
    lines(Kplus, col = 2)
    legend("topright", legend = c("K", expression(K["+"])),
           col = 1:2, lty = 1, box.lwd = 0)
}
}
