#' Mark correlation functions for homogeneous point patterns on linear networks.
#'
#' Mark correlation functions for homogeneous point patterns on linear networks.
#'
#' @usage \method{mcorr}{lpp}(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart",
#'  "isham", "stoyancov", "schlather"),
#' r = NULL,
#' method = c("density","loess"),
#' normalise = TRUE,
#' f = NULL,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class lpp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \eqn{r} at which the mark correlation function should be evaluated.
#' @param method Type of smoothing, either \code{density} or \code{loess}. See details.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param tol Tolerance used in the calculation of the conditional mean of the marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#' For a homogeneous point process \eqn{X} on a linear network, the \eqn{t_f}-correlation function \eqn{\kappa_{t_f}(r)} is given as
#' \deqn{
#' \kappa_{t_f}(r)
#'     =
#'     \frac{
#'     \mathbb{E} \left[
#'     t_f \left(
#'     m_x, m_y
#'     \right) \mid x, y \in X
#'     \right]
#'     }{
#'     c_{t_f}
#'     },
#'     \quad
#'     d(x,y)=r,
#' }
#' where \eqn{m_x, m_y} are the marks of \eqn{x, y \in X}, \eqn{c_{t_f}} is a normalizing factor, and \eqn{d(x,y)=r} is the shortest-path distance. Therefore, each mark correlation function is defined by a specific test function \eqn{t_f(m_x, m_y)} and its associated normalising factor \eqn{c_{t_f}}. Let \eqn{\mu_m} and \eqn{\sigma^2_m} be the mean and variance of marks, then, the list below gives different test functions \eqn{t_f} and their normalised factors \eqn{c_{t_f}}, following distinct available \code{ftype}.
#'
#' \describe{
#'   \item{variogram:}{
#'     \eqn{t_f(m_x, m_y) = \frac{1}{2}(m_x - m_y)^2},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#'   \item{stoyan:}{
#'     \eqn{t_f(m_x, m_y) = m_x m_y},
#'     \eqn{c_{t_f} = \mu^2_m}.
#'   }
#'   \item{rcorr:}{
#'     \eqn{t_f(m_x, m_y) = m_x},
#'     \eqn{c_{t_f} = \mu_m}.
#'   }
#'   \item{shimatani:}{
#'     \eqn{t_f(m_x, m_y) = (m_x - \mu_m)(m_y - \mu_m)},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#'   \item{beisbart:}{
#'     \eqn{t_f(m_x, m_y) = m_x + m_y},
#'     \eqn{c_{t_f} = 2 \mu_m}.
#'   }
#'   \item{isham:}{
#'     \eqn{t_f(m_x, m_y) = m_x m_y - \mu^2_m},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#'   \item{stoyancov:}{
#'     \eqn{t_f(m_x, m_y) = m_x m_y - \mu^2_m},
#'     \eqn{c_{t_f} = 1}.
#'   }
#'   \item{schlather:}{
#'     \eqn{t_f(m_x, m_y) = (m_x - \mu_m(r))(m_y - \mu_m(r))},
#'     \eqn{c_{t_f} = \sigma^2_m}.
#'   }
#' }
#' For \code{ftype="schlather"}, \eqn{\mu_m(r)} denotes the mean of the marks of all pairs of points whose pairwise distance lies within a tolerance \code{tol} of \eqn{r}.
#' We refer to Eckardt and Moradi (2024) for details of these mark correlation functions.
#'
#' Regarding the smoothing functions, if \code{method="density"}, the functions \code{\link[spatstat.univar]{unnormdensity}} will be called, and if \code{method="loess"}, the function \code{\link[stats]{loess}} will be called.
#'
#' If your \code{ftype} is not one of the defaults, then you need to give your test function \eqn{t_f(m_1, m_2)} using the argument \code{f}. In this case, \code{normalise} should be set as \code{FALSE}, as only the unnormalised version will be calculated. Depending on the form of the test function \eqn{t_f(m_1, m_2)}, one can manually compute the normalisation factor.
#'
#'
#' If the point patten \eqn{X} has multiple real-valued marks, the function estimates the mark correlation function for each mark individually. In such case, marks are given as a \code{data.frame} whose columns represents different marks. The functions checks which columns are numeric, and for those the mark correlation function will be computed.
#'
#'
#' @examples
#'  library(spatstat.linnet)
#'  library(spatstat.geom)
#'  X <- rpoislpp(10, simplenet)
#'  marks(X) <- runif(npoints(X), 1, 10)
#'  mcorr.lpp(X, ftype = "stoyan", method = "density")
#'
#' @references Eckardt, M., & Moradi, M. (2024). Marked spatial point processes: current state and extensions to point processes on linear networks. Journal of Agricultural, Biological and Environmental Statistics, 29(2), 346-378.
#' @return a data.frame which gives the estimated mark correlation function and the distance vector \eqn{r} at which the mark correlation function is estimated. If the point patten \eqn{X} has multiple real-valued marks, the estimated mark correlation function will be given for each mark. Name of columns will be the name of marks.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt
#'
#' @seealso \code{\link[markstat]{mcorr.ppp}}.


#' @import spatstat.linnet
#' @import spatstat.univar
#' @import spatstat.geom
#' @import spatstat.explore
#' @import spatstat.utils
#' @import stats
#' @export


mcorr.lpp <- function(X,
                      ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                      r = NULL,
                      method = c("density","loess"),
                      normalise = TRUE,
                      f = NULL,
                      tol = 0.01,
                      ...){

  if (all(class(X) != "lpp")) stop("object X should be of class lpp.")

  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")

  if (missing(method)) stop("smoothing method should be chosen.")

  m <- marks(X)

  if (any(class(m) == "hyperframe" | class(m) == "data.frame")){
    num_cols <- unlist(sapply(m, is.numeric))
    s <- which(num_cols)
    m <- as.data.frame(m)
    out <- list()
    for (i in 1:length(s)) {
      marks(X) <- as.numeric(m[,s[i]])
      out[[i]] <- mcorr.lpp(X, ftype = ftype, r = r, method = method,
                               normalise = normalise, f = f, tol = tol, ...)
    }

    r <- out[[1]]$r
    emps <- sapply(out, function(df) df$est)
    colnames(emps) <- names(s)
    finalout <- data.frame(r = r, emps)
    
    class(finalout) <- "mc"
    attr(finalout, "mtype") <- "real-valued"
    attr(finalout, "type") <- "global"
    attr(finalout, "ftype") <- ftype
    attr(finalout, "method") <- method
    attr(finalout, "normalise") <- normalise
    
    return(finalout)
  }

  if (is.null(f)) {
    if (ftype == "variogram") {
      f <- function(m1, m2, mu = NULL) 0.5 * ((m1 - m2)^2)
    } else if (ftype == "stoyan") {
      f <- function(m1, m2, mu = NULL) m1 * m2
    } else if (ftype == "rcorr") {
      f <- function(m1, m2, mu = NULL) m1
    } else if (ftype == "shimatani") {
      f <- function(m1, m2, mu = NULL) (m1 - mean(m)) * (m2 - mean(m))
    } else if (ftype == "beisbart") {
      f <- function(m1, m2, mu = NULL) m1 + m2
    } else if (ftype == "isham") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "stoyancov") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "schlather") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - mu * (m1 + m2) + mu^2
    } else {
      stop("Your ftype is not supported!")
    }
  } else {
    warning("Your given test function is not among the default ones; only unnormalised version will be calculated.")
  }
  


  n <- npoints(X)
  d <- pairdist.lpp(X)

  if(is.null(r)){
    L <- X$domain
    rmaxdefault <- 0.98 * boundingradius(L)
    if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
    W <- Window(L)
    breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
    r <- breaks$r
  }

  rmax <- max(r)


  df <- cbind(dist = as.vector(d),
              id.row = rep(c(1:n),each=n),
              id.col = rep(c(1:n),n))

  df.filter <- df[df[,1]< rmax & df[,1]>0,]

  m1 <- m[df.filter[,2]]
  m2 <- m[df.filter[,3]]

  if (ftype=="schlather"){
    df.filter <- cbind(df.filter,
                       mu = as.numeric(unlist(sapply(df.filter[,1], function(d) {
                         matched <- df.filter[,3][abs(df.filter[,1] - d) <= tol]
                         paste(mean(m[matched]), collapse = ",")
                       }))))
    mu <- df.filter[,4]
    dfvario <- data.frame(d=df.filter[,1], ff=(f(m1, m2, mu)))
  }else{
    dfvario <- data.frame(d=df.filter[,1], ff=(f(m1, m2)))
  }

  if(method=="density"){

    Kf <- unnormdensity(dfvario$d, weights = dfvario$ff,
                        from=min(r), to=max(r), n=length(r),
                        ...)$y

    ## smooth estimate of kappa_1
    K1 <- unnormdensity(dfvario$d, weights=rep(1,nrow(dfvario)),
                        from=min(r), to=max(r), n=length(r),
                        ...)$y

    Eff <- Kf/K1

  }else if(method=="loess"){

    lo <- loess(ff~d,data = dfvario,...)
    Eff <- predict(lo, newdata=data.frame(d=r))

  }else{
    stop("method should currently be either loess or density!!!")
  }


  if(normalise){
    if(ftype=="stoyan"){
      out <- Eff/(mean(m)^2)
    } else if(ftype=="variogram" | ftype=="isham" | ftype=="schlather" | ftype=="shimatani"){
      out <- Eff/var(m)
    }else if(ftype=="rcorr"){
      out <- Eff/mean(m)
    }else if(ftype=="Beisbart"){
      out <- Eff/(2*mean(m))
    }else if(ftype=="stoyancov"){
      out <- Eff
    }else{
      stop("your ftype is not supported!!")
    }
  }else{
    out <- Eff
  }

  out <- as.data.frame(cbind(r = r, est = out))
  
  if(ncol(out) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out) <- "mc"
  attr(out, "mtype") <- "real-valued"
  attr(out, "type") <- type
  attr(out, "ftype") <- ftype
  attr(out, "method") <- method
  attr(out, "normalise") <- normalise
  

  return(out)
}
