/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5IMprivate.h"
#include "H5LTprivate.h"

herr_t
H5IMmake_image_8bit(hid_t loc_id, const char *dset_name, hsize_t width, hsize_t height,
                    const unsigned char *buf)
{
    hsize_t dims[IMAGE8_RANK];

    
    if (dset_name == NULL)
        return -1;

    
    dims[0] = height;
    dims[1] = width;

    
    if (H5LTmake_dataset(loc_id, dset_name, IMAGE8_RANK, dims, H5T_NATIVE_UCHAR, buf) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "CLASS", IMAGE_CLASS) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "IMAGE_VERSION", IMAGE_VERSION) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "IMAGE_SUBCLASS", "IMAGE_INDEXED") < 0)
        return -1;

    return 0;
}

herr_t
H5IMmake_image_24bit(hid_t loc_id, const char *dset_name, hsize_t width, hsize_t height,
                     const char *interlace, const unsigned char *buf)
{
    hsize_t dims[IMAGE24_RANK];

    
    if (interlace == NULL)
        return -1;
    if (dset_name == NULL)
        return -1;

    

    if (strncmp(interlace, "INTERLACE_PIXEL", 15) == 0) {
        
        dims[0] = height;
        dims[1] = width;
        dims[2] = IMAGE24_RANK;
    }
    else if (strncmp(interlace, "INTERLACE_PLANE", 15) == 0) {
        
        dims[0] = IMAGE24_RANK;
        dims[1] = height;
        dims[2] = width;
    }
    else
        return -1;

    
    if (H5LTmake_dataset(loc_id, dset_name, IMAGE24_RANK, dims, H5T_NATIVE_UCHAR, buf) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "CLASS", IMAGE_CLASS) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "IMAGE_VERSION", IMAGE_VERSION) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "IMAGE_SUBCLASS", "IMAGE_TRUECOLOR") < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, dset_name, "INTERLACE_MODE", interlace) < 0)
        return -1;

    return 0;
}

static herr_t
find_palette(H5_ATTR_UNUSED hid_t loc_id, const char *name, H5_ATTR_UNUSED const H5A_info_t *ainfo,
             H5_ATTR_UNUSED void *op_data)
{
    int ret = H5_ITER_CONT;

    
    if (name == NULL)
        return -1;

    
    (void)loc_id;
    (void)ainfo;
    (void)op_data;

    
    if (strncmp(name, "PALETTE", 7) == 0)
        ret = H5_ITER_STOP;

    return ret;
}

herr_t
H5IM_find_palette(hid_t loc_id)
{
    return H5Aiterate2(loc_id, H5_INDEX_NAME, H5_ITER_INC, NULL, find_palette, NULL);
}

herr_t
H5IMget_image_info(hid_t loc_id, const char *dset_name, hsize_t *width, hsize_t *height, hsize_t *planes,
                   char *interlace, hssize_t *npals)
{
    hid_t       did = -1;
    hid_t       sid = -1;
    hsize_t     dims[IMAGE24_RANK];
    hid_t       aid  = -1;
    hid_t       asid = -1;
    hid_t       atid = -1;
    H5T_class_t aclass;
    int         has_pal;
    hid_t       has_attr;

    
    if (dset_name == NULL)
        return -1;
    if (interlace == NULL)
        return -1;

    
    *npals = 0;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((has_attr = H5Aexists(did, "INTERLACE_MODE")) < 0)
        goto out;

    
    if (has_attr > 0) {

        if ((aid = H5Aopen(did, "INTERLACE_MODE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if (H5Aread(aid, atid, interlace) < 0)
            goto out;

        if (H5Tclose(atid) < 0)
            goto out;

        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    if ((sid = H5Dget_space(did)) < 0)
        goto out;

    if (H5Sget_simple_extent_dims(sid, NULL, NULL) > IMAGE24_RANK)
        goto out;
    
    if (H5Sget_simple_extent_dims(sid, dims, NULL) < 0)
        goto out;

    

    if (has_attr > 0) {
        

        if (strncmp(interlace, "INTERLACE_PIXEL", 15) == 0) {
            
            *height = dims[0];
            *width  = dims[1];
            *planes = dims[2];
        }
        else if (strncmp(interlace, "INTERLACE_PLANE", 15) == 0) {
            
            *planes = dims[0];
            *height = dims[1];
            *width  = dims[2];
        }
        else
            return -1;
    }
    else {
        
        *height = dims[0];
        *width  = dims[1];
        *planes = 1;
    }

    
    if (H5Sclose(sid) < 0)
        goto out;

    

    
    has_pal = H5IM_find_palette(did);

    if (has_pal == 1) {

        if ((aid = H5Aopen(did, "PALETTE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if ((aclass = H5Tget_class(atid)) < 0)
            goto out;

        

        if (aclass == H5T_REFERENCE) {

            

            if ((asid = H5Aget_space(aid)) < 0)
                goto out;

            *npals = H5Sget_simple_extent_npoints(asid);

            if (H5Sclose(asid) < 0)
                goto out;

        } 

        if (H5Tclose(atid) < 0)
            goto out;

        
        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    if (H5Dclose(did) < 0)
        goto out;

    return 0;

out:
    if (did > 0)
        H5Dclose(did);
    if (aid > 0)
        H5Aclose(aid);
    if (asid > 0)
        H5Sclose(asid);
    if (atid > 0)
        H5Tclose(atid);
    return -1;
}

herr_t
H5IMread_image(hid_t loc_id, const char *dset_name, unsigned char *buf)
{
    hid_t did;

    
    if (dset_name == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if (H5Dread(did, H5T_NATIVE_UCHAR, H5S_ALL, H5S_ALL, H5P_DEFAULT, buf) < 0)
        goto out;

    
    if (H5Dclose(did))
        return -1;

    return 0;

out:
    H5Dclose(did);
    return -1;
}

herr_t
H5IMmake_palette(hid_t loc_id, const char *pal_name, const hsize_t *pal_dims, const unsigned char *pal_data)

{

    int has_pal;

    
    if (pal_name == NULL)
        return -1;

    
    has_pal = H5LTfind_dataset(loc_id, pal_name);

    
    if (has_pal == 1)
        return 0;

    
    if (H5LTmake_dataset(loc_id, pal_name, 2, pal_dims, H5T_NATIVE_UCHAR, pal_data) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, pal_name, "CLASS", PALETTE_CLASS) < 0)
        return -1;

    
    if (H5LTset_attribute_string(loc_id, pal_name, "PAL_VERSION", "1.2") < 0)
        return -1;

    return 0;
}

herr_t
H5IMlink_palette(hid_t loc_id, const char *image_name, const char *pal_name)

{
    hid_t       did;
    hid_t       atid = -1;
    hid_t       aid  = -1;
    hid_t       asid = -1;
    hobj_ref_t  ref;    
    hobj_ref_t *refbuf; 
    hssize_t    n_refs;
    hsize_t     dim_ref;
    htri_t      ok_pal;

    
    if (image_name == NULL)
        return -1;
    if (pal_name == NULL)
        return -1;

    

    
    if ((did = H5Dopen2(loc_id, image_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((ok_pal = H5Aexists(did, "PALETTE")) < 0)
        goto out;

    
    if (ok_pal == 0) {
        if ((asid = H5Screate(H5S_SCALAR)) < 0)
            goto out;

        
        if ((atid = H5Tcopy(H5T_STD_REF_OBJ)) < 0)
            goto out;

        
        if ((aid = H5Acreate2(did, "PALETTE", atid, asid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        if (H5Rcreate(&ref, loc_id, pal_name, H5R_OBJECT, (hid_t)-1) < 0)
            goto out;

        
        if (H5Awrite(aid, atid, &ref) < 0)
            goto out;

        
        if (H5Sclose(asid) < 0)
            goto out;
        if (H5Tclose(atid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    else if (ok_pal > 0) {
        if ((aid = H5Aopen(did, "PALETTE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if (H5Tget_class(atid) < 0)
            goto out;

        
        if ((asid = H5Aget_space(aid)) < 0)
            goto out;

        n_refs = H5Sget_simple_extent_npoints(asid);

        dim_ref = (hsize_t)n_refs + 1;

        refbuf = (hobj_ref_t *)malloc(sizeof(hobj_ref_t) * (size_t)dim_ref);

        if (H5Aread(aid, atid, refbuf) < 0)
            goto out;

        
        if (H5Adelete(did, "PALETTE") < 0)
            goto out;

        
        if (H5Rcreate(&ref, loc_id, pal_name, H5R_OBJECT, (hid_t)-1) < 0)
            goto out;

        refbuf[n_refs] = ref;

        
        if (H5Sclose(asid) < 0)
            goto out;

        if ((asid = H5Screate_simple(1, &dim_ref, NULL)) < 0)
            goto out;

        
        if (H5Aclose(aid) < 0)
            goto out;

        if ((aid = H5Acreate2(did, "PALETTE", atid, asid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        if (H5Awrite(aid, atid, refbuf) < 0)
            goto out;

        
        if (H5Sclose(asid) < 0)
            goto out;
        if (H5Tclose(atid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;

        free(refbuf);

    } 

    
    if (H5Dclose(did) < 0)
        return -1;

    return 0;

out:
    H5Dclose(did);
    H5Sclose(asid);
    H5Tclose(atid);
    H5Aclose(aid);
    return -1;
}

herr_t
H5IMunlink_palette(hid_t loc_id, const char *image_name, const char *pal_name)
{
    hid_t       did;
    hid_t       atid;
    hid_t       aid;
    H5T_class_t aclass;
    htri_t      ok_pal;
    int         has_pal;

    
    if (image_name == NULL)
        return -1;
    if (pal_name == NULL)
        return -1;

    
    has_pal = H5LTfind_dataset(loc_id, pal_name);

    
    if (has_pal == 0)
        return -1;

    

    
    if ((did = H5Dopen2(loc_id, image_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((ok_pal = H5Aexists(did, "PALETTE")) < 0)
        goto out;

    
    if (ok_pal == 0)
        goto out;
    else if (ok_pal > 0) {
        
        if ((aid = H5Aopen(did, "PALETTE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if ((aclass = H5Tget_class(atid)) < 0)
            goto out;

        
        if (aclass == H5T_REFERENCE) {
            
            if (H5Adelete(did, "PALETTE") < 0)
                goto out;

        } 

        if (H5Tclose(atid) < 0)
            goto out;

        
        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    if (H5Dclose(did) < 0)
        return -1;

    return 0;

out:
    H5Dclose(did);
    return -1;
}

herr_t
H5IMget_npalettes(hid_t loc_id, const char *image_name, hssize_t *npals)
{
    hid_t       did;
    hid_t       atid;
    hid_t       aid;
    hid_t       asid;
    H5T_class_t aclass;
    int         has_pal;

    
    if (image_name == NULL)
        return -1;

    
    *npals = 0;

    
    if ((did = H5Dopen2(loc_id, image_name, H5P_DEFAULT)) < 0)
        return -1;

    
    has_pal = H5IM_find_palette(did);

    if (has_pal == 1) {

        if ((aid = H5Aopen(did, "PALETTE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if ((aclass = H5Tget_class(atid)) < 0)
            goto out;

        

        if (aclass == H5T_REFERENCE) {
            if ((asid = H5Aget_space(aid)) < 0)
                goto out;

            *npals = H5Sget_simple_extent_npoints(asid);

            if (H5Sclose(asid) < 0)
                goto out;

        } 

        if (H5Tclose(atid) < 0)
            goto out;

        
        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    if (H5Dclose(did) < 0)
        return -1;

    return 0;

out:
    H5Dclose(did);
    return -1;
}

herr_t
H5IMget_palette_info(hid_t loc_id, const char *image_name, int pal_number, hsize_t *pal_dims)
{
    hid_t       did;
    int         has_pal;
    hid_t       atid = -1;
    hid_t       aid;
    hid_t       asid = -1;
    hssize_t    n_refs;
    hsize_t     dim_ref;
    hobj_ref_t *refbuf; 
    hid_t       pal_id;
    hid_t       pal_space_id;
    hsize_t     pal_maxdims[2];

    
    if (image_name == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, image_name, H5P_DEFAULT)) < 0)
        return -1;

    
    has_pal = H5IM_find_palette(did);

    if (has_pal == 1) {
        if ((aid = H5Aopen(did, "PALETTE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if (H5Tget_class(atid) < 0)
            goto out;

        
        if ((asid = H5Aget_space(aid)) < 0)
            goto out;

        n_refs = H5Sget_simple_extent_npoints(asid);

        dim_ref = (hsize_t)n_refs;

        refbuf = (hobj_ref_t *)malloc(sizeof(hobj_ref_t) * (size_t)dim_ref);

        if (H5Aread(aid, atid, refbuf) < 0)
            goto out;

        
        if ((pal_id = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &refbuf[pal_number])) < 0)
            goto out;

        if ((pal_space_id = H5Dget_space(pal_id)) < 0)
            goto out;

        if (H5Sget_simple_extent_ndims(pal_space_id) < 0)
            goto out;

        if (H5Sget_simple_extent_dims(pal_space_id, pal_dims, pal_maxdims) < 0)
            goto out;

        
        if (H5Dclose(pal_id) < 0)
            goto out;
        if (H5Sclose(pal_space_id) < 0)
            goto out;
        if (H5Sclose(asid) < 0)
            goto out;
        if (H5Tclose(atid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        free(refbuf);
    }

    
    if (H5Dclose(did) < 0)
        return -1;

    return 0;

out:
    H5Dclose(did);
    H5Sclose(asid);
    H5Tclose(atid);
    H5Aclose(aid);
    return -1;
}

herr_t
H5IMget_palette(hid_t loc_id, const char *image_name, int pal_number, unsigned char *pal_data)
{
    hid_t       did;
    int         has_pal;
    hid_t       atid = -1;
    hid_t       aid;
    hid_t       asid = -1;
    hssize_t    n_refs;
    hsize_t     dim_ref;
    hobj_ref_t *refbuf; 
    hid_t       pal_id;

    
    if (image_name == NULL)
        return -1;
    if (pal_data == NULL)
        return -1;

    
    if ((did = H5Dopen2(loc_id, image_name, H5P_DEFAULT)) < 0)
        return -1;

    
    has_pal = H5IM_find_palette(did);

    if (has_pal == 1) {
        if ((aid = H5Aopen(did, "PALETTE", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        if (H5Tget_class(atid) < 0)
            goto out;

        
        if ((asid = H5Aget_space(aid)) < 0)
            goto out;

        n_refs = H5Sget_simple_extent_npoints(asid);

        dim_ref = (hsize_t)n_refs;

        refbuf = (hobj_ref_t *)malloc(sizeof(hobj_ref_t) * (size_t)dim_ref);

        if (H5Aread(aid, atid, refbuf) < 0)
            goto out;

        
        if ((pal_id = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &refbuf[pal_number])) < 0)
            goto out;

        
        if (H5Dread(pal_id, H5Dget_type(pal_id), H5S_ALL, H5S_ALL, H5P_DEFAULT, pal_data) < 0)
            goto out;

        
        if (H5Dclose(pal_id) < 0)
            goto out;
        if (H5Sclose(asid) < 0)
            goto out;
        if (H5Tclose(atid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        free(refbuf);
    }

    
    if (H5Dclose(did) < 0)
        return -1;

    return 0;

out:
    H5Dclose(did);
    H5Sclose(asid);
    H5Tclose(atid);
    H5Aclose(aid);
    return -1;
}

herr_t
H5IMis_image(hid_t loc_id, const char *dset_name)
{
    hid_t   did;
    htri_t  has_class;
    hid_t   atid;
    hid_t   aid = -1;
    char   *attr_data;    
    hsize_t storage_size; 
    herr_t  ret;

    
    if (dset_name == NULL)
        return -1;

    
    ret = -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((has_class = H5Aexists(did, "CLASS")) < 0)
        goto out;

    if (has_class == 0) {
        H5Dclose(did);
        return 0;
    }
    else {

        if ((aid = H5Aopen(did, "CLASS", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        
        if (H5T_STRING != H5Tget_class(atid))
            goto out;

        
        if (H5T_STR_NULLTERM != H5Tget_strpad(atid))
            goto out;

        
        if ((storage_size = H5Aget_storage_size(aid)) == 0)
            goto out;

        attr_data = (char *)malloc((size_t)storage_size * sizeof(char) + 1);
        if (attr_data == NULL)
            goto out;

        if (H5Aread(aid, atid, attr_data) < 0)
            goto out;

        if (strncmp(attr_data, IMAGE_CLASS, MIN(strlen(IMAGE_CLASS), strlen(attr_data))) == 0)
            ret = 1;
        else
            ret = 0;

        free(attr_data);

        if (H5Tclose(atid) < 0)
            goto out;

        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    if (H5Dclose(did) < 0)
        return -1;

    return ret;

out:
    H5Dclose(did);
    return -1;
}

herr_t
H5IMis_palette(hid_t loc_id, const char *dset_name)
{
    hid_t   did;
    htri_t  has_class;
    hid_t   atid;
    hid_t   aid = -1;
    char   *attr_data;    
    hsize_t storage_size; 
    herr_t  ret;

    
    if (dset_name == NULL)
        return -1;

    
    ret = -1;

    
    if ((did = H5Dopen2(loc_id, dset_name, H5P_DEFAULT)) < 0)
        return -1;

    
    if ((has_class = H5Aexists(did, "CLASS")) < 0)
        goto out;

    if (has_class == 0) {
        H5Dclose(did);
        return 0;
    }
    else {

        if ((aid = H5Aopen(did, "CLASS", H5P_DEFAULT)) < 0)
            goto out;

        if ((atid = H5Aget_type(aid)) < 0)
            goto out;

        
        if (H5T_STRING != H5Tget_class(atid))
            goto out;

        
        if (H5T_STR_NULLTERM != H5Tget_strpad(atid))
            goto out;

        
        if ((storage_size = H5Aget_storage_size(aid)) == 0)
            goto out;

        attr_data = (char *)malloc((size_t)storage_size * sizeof(char) + 1);
        if (attr_data == NULL)
            goto out;

        if (H5Aread(aid, atid, attr_data) < 0)
            goto out;

        if (strncmp(attr_data, PALETTE_CLASS, MIN(strlen(PALETTE_CLASS), strlen(attr_data))) == 0)
            ret = 1;
        else
            ret = 0;

        free(attr_data);

        if (H5Tclose(atid) < 0)
            goto out;

        if (H5Aclose(aid) < 0)
            goto out;
    }

    
    if (H5Dclose(did) < 0)
        return -1;

    return ret;

out:
    H5Dclose(did);
    return -1;
}
