% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom-edges.R
\name{geom_edges}
\alias{geom_edges}
\title{Draw the edges of a network.}
\usage{
geom_edges(
  mapping = NULL,
  data = NULL,
  position = "identity",
  arrow = NULL,
  curvature = 0,
  angle = 90,
  ncp = 5,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  ...
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{position}{A position adjustment to use on the data for this layer. This
can be used in various ways, including to prevent overplotting and
improving the display. The \code{position} argument accepts the following:
\itemize{
\item The result of calling a position function, such as \code{position_jitter()}.
This method allows for passing extra arguments to the position.
\item A string naming the position adjustment. To give the position as a
string, strip the function name of the \code{position_} prefix. For example,
to use \code{position_jitter()}, give the position as \code{"jitter"}.
\item For more information and other ways to specify the position, see the
\link[ggplot2:layer_positions]{layer position} documentation.
}}

\item{arrow}{specification for arrow heads, as created by \code{\link[grid:arrow]{grid::arrow()}}.}

\item{curvature}{A numeric value giving the amount of curvature.
    Negative values produce left-hand curves, positive values
    produce right-hand curves, and zero produces a straight line.}

\item{angle}{A numeric value between 0 and 180,
    giving an amount to skew the control
    points of the curve.  Values less than 90 skew the curve towards
    the start point and values greater than 90 skew the curve
    towards the end point.}

\item{ncp}{The number of control points used to draw the curve.
    More control points creates a smoother curve.}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}

\item{...}{Other arguments passed on to \code{\link[ggplot2:layer]{layer()}}'s \code{params} argument. These
arguments broadly fall into one of 4 categories below. Notably, further
arguments to the \code{position} argument, or aesthetics that are required
can \emph{not} be passed through \code{...}. Unknown arguments that are not part
of the 4 categories below are ignored.
\itemize{
\item Static aesthetics that are not mapped to a scale, but are at a fixed
value and apply to the layer as a whole. For example, \code{colour = "red"}
or \code{linewidth = 3}. The geom's documentation has an \strong{Aesthetics}
section that lists the available options. The 'required' aesthetics
cannot be passed on to the \code{params}. Please note that while passing
unmapped aesthetics as vectors is technically possible, the order and
required length is not guaranteed to be parallel to the input data.
\item When constructing a layer using
a \verb{stat_*()} function, the \code{...} argument can be used to pass on
parameters to the \code{geom} part of the layer. An example of this is
\code{stat_density(geom = "area", outline.type = "both")}. The geom's
documentation lists which parameters it can accept.
\item Inversely, when constructing a layer using a
\verb{geom_*()} function, the \code{...} argument can be used to pass on parameters
to the \code{stat} part of the layer. An example of this is
\code{geom_area(stat = "density", adjust = 0.5)}. The stat's documentation
lists which parameters it can accept.
\item The \code{key_glyph} argument of \code{\link[ggplot2:layer]{layer()}} may also be passed on through
\code{...}. This can be one of the functions described as
\link[ggplot2:draw_key]{key glyphs}, to change the display of the layer in the legend.
}}
}
\description{
All arguments to this geom are identical to those of
\code{\link[ggplot2]{geom_segment}}, including \code{arrow}, which is useful
to plot directed networks in conjunction with the \code{arrow.gap} argument
of \code{\link{fortify.network}}. The \code{curvature}, \code{angle} and
\code{ncp} arguments of \code{\link[ggplot2]{geom_curve}} are also available:
if \code{curvature} is set to any value above \code{0} (the default), the
edges produced by \code{geom_edges} will be curved.
}
\examples{
if (require(network) && require(sna)) {

  # rerun if the example does not produce reciprocated ties
  n <- network(rgraph(10, tprob = 0.2), directed = TRUE)

  # just edges
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(size = 1, colour = "steelblue") +
    theme_blank()

  # with nodes
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(size = 1, colour = "steelblue") +
    geom_nodes(size = 3, colour = "steelblue") +
    theme_blank()

  # with arrows
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(
      size = 1, colour = "steelblue",
      arrow = arrow(length = unit(0.5, "lines"), type = "closed")
    ) +
    geom_nodes(size = 3, colour = "steelblue") +
    theme_blank()

  # with curvature
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(
      size = 1, colour = "steelblue", curvature = 0.15,
      arrow = arrow(length = unit(0.5, "lines"), type = "closed")
    ) +
    geom_nodes(size = 3, colour = "steelblue") +
    theme_blank()

  # arbitrary categorical edge attribute
  e <- sample(letters[ 1:2 ], network.edgecount(n), replace = TRUE)
  set.edge.attribute(n, "type", e)
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(aes(linetype = type),
      size = 1, curvature = 0.15,
      arrow = arrow(length = unit(0.5, "lines"), type = "closed")
    ) +
    geom_nodes(size = 3, colour = "steelblue") +
    theme_blank()

  # arbitrary numeric edge attribute (signed network)
  e <- sample(-2:2, network.edgecount(n), replace = TRUE)
  set.edge.attribute(n, "weight", e)
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(aes(colour = weight),
      curvature = 0.15,
      arrow = arrow(length = unit(0.5, "lines"), type = "closed")
    ) +
    geom_nodes(size = 3, colour = "grey50") +
    scale_colour_gradient(low = "steelblue", high = "tomato") +
    theme_blank()

  # draw only a subset of all edges
  positive_weight <- function(x) {
    x[ x$weight >= 0, ]
  }
  ggplot(n, aes(x, y, xend = xend, yend = yend)) +
    geom_edges(aes(colour = weight), data = positive_weight) +
    geom_nodes(size = 4, colour = "grey50") +
    scale_colour_gradient(low = "gold", high = "tomato") +
    theme_blank()
}

}
