#' @title Canopy Temperature Depression (CTD)
#' @description This function computes canopy temperature depression (CTD). For more details see Watson (1958) <doi:10.1093/oxfordjournals.aob.a083596>.
#' @param Gen Character vector of genotype IDs.
#' @param amb.temp Numeric scalar; ambient temperature (°C) (user must define this input).
#' @param CT_PI Numeric vector; canopy temperature (°C) at phase I (PI).
#' @return A list of output columns:
#' \itemize{
#'  \item \code{Gen}: Character vector of genotype IDs.
#'  \item \code{CTD}: Canopy temperature depression (CTD) values for respective genotypes.
#'  }
#' @references Watson, D.J. (1958). <doi:10.1093/oxfordjournals.aob.a083596>.
#' @examples
#' # Creating a sample dataset
#' df <- data.frame(
#' Gen = c("V1","V2","V3"),
#' CT_PI   = c(26.7,31.4,28.4)
#' )
#' ctd(df$Gen, amb.temp = 34.2, CT_PI = df$CT_PI)
#'
#' @export
ctd <- function(Gen, amb.temp, CT_PI){
  CTD <- round(amb.temp - CT_PI, 4)
  return(data.frame(Gen, CTD, row.names = NULL))
}
