#' @title Computation of All Stress Indices at Once
#' @description
#' A convenience function that returns a data frame with 11 stress indices—Stress Tolerance (TOL), Stress Tolerance Index (STI), Stress Susceptibility Percentage Index (SSPI), Yield Index (YI), Yield Stability Index (YSI), Relative Stress Index (RSI), Mean Productivity (MP), Geometric Mean Productivity (GMP), Harmonic Mean (HM), Mean Relative Performance (MRP), and Percent Yield Reduction (PYR)—for the same given inputs (Lamba et al., 2023; <doi:10.1038/s41598-023-37634-8>).
#'
#' @param Gen Character vector of Genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @param YMN,YMS Optional numeric scalars for environment means. If NULL, computed.
#' @param include_inputs Logical; if TRUE returns a list with \code{YMN}, \code{YMS}, and \code{all}.
#' @param name_vectors_by_gen Ignored; kept for backward compatibility.
#' @return If \code{include_inputs=TRUE}, a list with \code{YMN}, \code{YMS}, and \code{all} data frame.
#' @references Lamba, K., Kumar, M., Singh, V. et al. (2023). <doi:10.1038/s41598-023-37634-8>.
#' @examples
#' out <- all_indices(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5), YS=c(7,5,3)
#' )
#' print(out)
#' @export
all_indices <- function(Gen, YN, YS, YMN = NULL, YMS = NULL,
                        include_inputs = TRUE, name_vectors_by_gen = TRUE) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  if (is.null(YMN)) YMN <- mean(YN, na.rm = TRUE)
  if (is.null(YMS)) YMS <- mean(YS, na.rm = TRUE)

  TOL <- YN - YS
  denom_sti <- (YMN^2)
  STI <- (YN * YS) / denom_sti
  STI[denom_sti == 0] <- NA_real_

  denom_sspi <- 2 * YMN
  SSPI <- ((YN - YS) / denom_sspi) * 100
  SSPI[denom_sspi == 0] <- NA_real_

  YI <- YS / YMS
  YI[YMS == 0] <- NA_real_

  YSI <- YS / YN
  YSI[YN == 0] <- NA_real_

  ratio_env <- YMS / YMN
  RSI <- (YN / YS) / ratio_env
  bad_rsi <- (YS == 0) | (ratio_env == 0) | !is.finite(ratio_env)
  RSI[bad_rsi] <- NA_real_

  MP <- (YN + YS) / 2

  prod <- YN * YS
  prod[prod < 0] <- NA_real_
  GMP <- sqrt(prod)

  denom_hm <- YN + YS
  HM <- 2 * (YN * YS) / denom_hm
  HM[denom_hm == 0] <- NA_real_

  p1 <- YS / YMS; p1[YMS == 0] <- NA_real_
  p2 <- YN / YMN; p2[YMN == 0] <- NA_real_
  MRP <- p1 + p2

  PYR <- ((YN - YS) / YN) * 100
  PYR[YN == 0] <- NA_real_

  all <- data.frame(
    Gen = as.character(Gen),
    TOL = TOL, STI = STI, SSPI = SSPI, YI = YI, YSI = YSI, RSI = RSI,
    MP = MP, GMP = GMP, HM = HM, MRP = MRP, PYR = PYR
  )

  if (isTRUE(include_inputs)) {
    out <- list(YMN = YMN, YMS = YMS, all = all)
  } else {
    out <- all
  }
  out
}
