#' Comprehensive Analysis of Factorial Block Designs
#'
#' FactChar() performs an extensive diagnostic and characterization of
#' mixed-level factorial designs arranged in blocks.
#'
#' The function computes, among other things:
#' - incidence matrices
#' - C-matrix and its eigenstructure
#' - A-, D-, E-, and MV-efficiencies
#' - balance checks for model effects
#' - OFS (Orthogonal Factorial Structure) checks
#' - Hamming distance based measures
#' - discrepancy indices (DD)
#' - B-criterion
#' - Es^2 (Xu and Wu) for mixed-level designs
#' - J2-distance and J2-optimization
#' - Phi_p optimality values
#' - symmetry checks for universal optimality
#'
#' The function validates the user-supplied blocks so that all treatment
#' combinations are consistent with the supplied factor_levels.
#'
#' @section Input validation:
#' If factor_levels = c(s1, s2, ..., sm), then each treatment label inside
#' the blocks list must:
#' - be a character string of length m
#' - use digits 0, 1, ..., s_i - 1 in position i
#' - match one of the prod(s_i) possible treatment combinations
#'
#' If the user enters any treatment combination that does not match
#' factor_levels, the function stops with the message:
#'
#' "treatment combination is not match with your factor_levels"
#'
#' @param factor_levels Integer vector giving the number of levels of each
#'   factor, in order. For example, factor_levels = c(3, 4) means factor 1 has
#'   3 levels coded 0, 1, 2 and factor 2 has 4 levels coded 0, 1, 2, 3.
#'
#' @param blocks A named list of blocks. Each element is a character vector of
#'   treatment labels. A treatment label is a string of digits, one digit per
#'   factor, using 0-based coding that matches factor_levels.
#'
#'   Example:
#'   \code{
#'   blocks <- list(
#'     B1 = c("00", "10", "20"),
#'     B2 = c("01", "11", "21"),
#'     B3 = c("02", "12", "22"),
#'     B4 = c("03", "13", "23")
#'   )
#'   }
#'
#' @param verbose Logical; if TRUE (default) a detailed summary is printed
#'   to the console. If FALSE, all informational output is suppressed and
#'   only the result object is returned (invisibly).
#'
#' @return
#' Invisibly returns a list containing the main diagnostic objects:
#' incidence structure, C-matrix and eigenvalues, efficiency measures,
#' OFS summary, discrepancy and distance measures, J2-criteria, Phi_p
#' values, and symmetry diagnostics. If \code{verbose = TRUE}, a detailed
#' summary is printed to the console.
#'
#' @details
#' Internally, the function:
#' - builds the treatment list implied by factor_levels
#' - validates that all user-specified treatments in blocks are compatible
#'   with factor_levels
#' - builds the incidence matrix N
#' - computes the C-matrix and its eigenvalues and rank
#' - evaluates estimability and balance of factorial effects
#' - computes OFS (Orthogonal Factorial Structure) measures
#' - computes various discrepancy and distance measures (Hamming distance,
#'   discrete discrepancy DD)
#' - computes MMA (Moment Matrix Analysis: K-vector)
#' - computes MA or GMA measures (A-vector)
#' - computes Es^2 for mixed-level designs
#' - computes J2-distance, J2-based efficiencies, and a lower bound
#' - computes Phi_p values, trace(C^2), and distance from equal eigenvalues
#' - checks symmetry conditions that are sufficient for universal optimality
#'
#' The implementation includes helper routines based on methods from
#' Das, Dean, Stufken, Wu, Hamada, Xu, Wu, Cheng, and Mukerjee, among others.
#'
#' @examples
#' ## A valid 3 x 4 factorial in 4 blocks
#' factor_levels <- c(3, 4)
#'
#' blocks <- list(
#'   B1 = c("00", "10", "20"),
#'   B2 = c("01", "11", "21"),
#'   B3 = c("02", "12", "22"),
#'   B4 = c("03", "13", "23")
#' )
#'
#' out <- FactChar(factor_levels, blocks, verbose = FALSE)
#' str(out)
#'
#' @references
#' Xu, H. and Wu, C. F. J. (2001).
#'   Generalized minimum aberration for asymmetrical fractional factorial designs.
#'   Annals of Statistics, 29, 1066-1077.
#'
#' Qin, H. and Ai, M. (2007).
#'   A note on the connection between uniformity and generalized minimum aberration.
#'   Statistical Papers, 48, 491-502.
#'
#' Gupta, S. C. (1983).
#'   Some new methods for constructing block designs having orthogonal factorial structure.
#'   Journal of the Royal Statistical Society, Series B (Methodological), 45, 297-307.
#' @import MASS
#' @import Matrix
#' @importFrom stats as.formula contr.helmert contr.sum
#' @importFrom stats "contrasts<-" model.matrix rnorm terms
#' @importFrom utils combn
#' @export
FactChar <- function(factor_levels, blocks, verbose = TRUE) {

  mpinv <- function(A, tol = 1e-8) {
    sv <- svd(A)
    d <- ifelse(sv$d > tol, 1 / sv$d, 0)
    sv$v %*% diag(d, nrow = length(d)) %*% t(sv$u)
  }
  ensure_col   <- function(x, v) matrix(as.numeric(x), nrow = v, ncol = 1)
  is_zero_vec  <- function(x, tol = 1e-10) all(abs(x) < tol)

  build_incidence <- function(levels, blocks) {
    grids <- do.call(
      expand.grid,
      lapply(rev(levels), function(s) 0:(s - 1))
    )
    grids <- grids[, rev(seq_len(ncol(grids)))]
    trt_labels <- apply(grids, 1, paste0, collapse = "")

    v <- length(trt_labels)
    b <- length(blocks)
    N <- matrix(0, nrow = v, ncol = b)
    rownames(N) <- trt_labels
    colnames(N) <- paste0("B", seq_len(b))

    ## Validation
    user_treatments <- unique(unlist(blocks))

    wrong_len <- user_treatments[nchar(user_treatments) != length(levels)]
    if (length(wrong_len) > 0) {
      stop(
        sprintf(
          "Treatment combination(s) %s are not valid: expected %d digits per treatment for factor_levels = c(%s).",
          paste(shQuote(wrong_len), collapse = ", "),
          length(levels),
          paste(levels, collapse = ", ")
        )
      )
    }

    invalid <- setdiff(user_treatments, trt_labels)
    if (length(invalid) > 0) {
      stop(
        sprintf(
          "Treatment combination(s) %s do not match your factor_levels = c(%s).\nPlease correct blocks: only combinations from {%s} are allowed.",
          paste(shQuote(invalid), collapse = ", "),
          paste(levels, collapse = ", "),
          paste(trt_labels, collapse = ", ")
        )
      )
    }

    for (h in seq_len(b)) {
      for (trt in blocks[[h]]) {
        pos <- match(trt, trt_labels)
        if (is.na(pos)) {
          stop(sprintf(
            "Internal error: treatment '%s' not found. Check factor_levels and blocks.",
            trt
          ))
        }
        N[pos, h] <- N[pos, h] + 1
      }
    }

    list(N = N, treatments = trt_labels, grids = grids)
  }

  compute_Cmatrix <- function(N) {
    v <- nrow(N); b <- ncol(N)
    r <- rowSums(N); k <- colSums(N)
    Rdel <- diag(as.numeric(r), v, v)
    Kdel <- diag(as.numeric(k), b, b)
    C <- Rdel - N %*% solve(Kdel) %*% t(N)
    list(C = C, r = r, k = k)
  }

  build_contrast_matrix <- function(grids) {
    df <- as.data.frame(grids, stringsAsFactors = TRUE)
    for (nm in names(df)) {
      df[[nm]] <- as.factor(df[[nm]])
      contrasts(df[[nm]]) <- contr.sum(nlevels(df[[nm]]))
    }
    form <- paste0("~ (", paste(names(df), collapse = "*"), ") - 1")
    mm <- model.matrix(as.formula(form), data = df)
    assign_vec  <- attr(mm, "assign")
    term_labels <- attr(terms(as.formula(form), data = df), "term.labels")
    effects <- sapply(assign_vec, function(a) if (a == 0) "(Intercept)" else term_labels[a])
    attr(mm, "effect") <- effects
    mm
  }

  compute_J2 <- function(f_matrix, weights = NULL) {
    n <- nrow(f_matrix)
    m <- ncol(f_matrix)
    if (is.null(weights)) weights <- rep(1, m)
    if (length(weights) != m) stop("weights length must equal number of factors")
    J2_sum <- 0
    for (i in 1:(n - 1)) {
      row_i <- f_matrix[i, , drop = FALSE]
      for (j in (i + 1):n) {
        delta_ij <- sum(weights * (row_i == f_matrix[j, ]))
        J2_sum <- J2_sum + (delta_ij^2)
      }
    }
    J2_sum
  }

  mpinv2 <- mpinv

  is_estimable <- function(C, l, tol = 1e-8) {
    v <- nrow(C); lcol <- ensure_col(l, v); Cplus <- mpinv2(C)
    diff <- C %*% Cplus %*% lcol - lcol
    all(abs(diff) < tol)
  }

  contrast_var_factor <- function(C, l) {
    v <- nrow(C); lcol <- ensure_col(l, v)
    as.numeric(t(lcol) %*% mpinv2(C) %*% lcol)
  }

  evaluate_balance <- function(C, mm, tol = 1e-8, var_tol = 1e-8) {
    effects <- unique(attr(mm, "effect")); results <- list()
    for (eff in effects) {
      idx  <- which(attr(mm, "effect") == eff)
      cols <- mm[, idx, drop = FALSE]
      est_flags <- logical(ncol(cols))
      vars      <- rep(NA_real_, ncol(cols))
      for (j in seq_len(ncol(cols))) {
        l <- cols[, j]
        if (is_zero_vec(l, tol)) next
        lnorm <- l / sqrt(sum(l^2))
        est_flags[j] <- is_estimable(C, lnorm, tol)
        if (est_flags[j]) vars[j] <- contrast_var_factor(C, lnorm)
      }
      if (all(!est_flags)) {
        balanced <- TRUE; reason <- "no contrast estimable"
      } else if (all(est_flags)) {
        if ((max(vars, na.rm = TRUE) - min(vars, na.rm = TRUE)) < var_tol) {
          balanced <- TRUE; reason <- "equal variance"
        } else {
          balanced <- FALSE; reason <- "unequal variance"
        }
      } else {
        balanced <- FALSE; reason <- "partial estimability"
      }
      results[[eff]] <- list(balanced = balanced, reason = reason)
    }
    results
  }

  check_design_OFS <- function(factor_levels, blocks) {
    decode_treatment <- function(trt) as.numeric(strsplit(trt, "")[[1]])
    code_levels <- function(s) {
      if (s %% 2 == 1) seq(-(s - 1) / 2, (s - 1) / 2, by = 1)
      else seq(-(s - 1), (s - 1), by = 2)
    }
    all_treatments <- unique(unlist(blocks))
    if (length(all_treatments) == 0) return(list(OFS = FALSE, mean_corr = NA))
    m <- nchar(all_treatments[1])
    design_data <- data.frame()
    for (b in seq_along(blocks)) {
      block <- blocks[[b]]
      for (t in block) {
        factors <- decode_treatment(t)
        design_data <- rbind(design_data, cbind(Block = paste0("B", b), t(factors)))
      }
    }
    colnames(design_data)[-1] <- paste0("F", 1:m)
    coded_data <- design_data
    for (i in 1:m) {
      s <- factor_levels[i]
      if (is.na(s) || s < 1) return(list(OFS = FALSE, mean_corr = NA))
      code_map <- code_levels(s)
      map_df <- data.frame(orig = 0:(s - 1), coded = code_map)
      coded_data[[paste0("F", i)]] <- sapply(design_data[[paste0("F", i)]],
                                             function(x) map_df$coded[map_df$orig == x])
    }
    form_str <- paste("~ (", paste(colnames(coded_data)[-1], collapse = " + "), ")^", m, sep = "")
    form <- as.formula(form_str)
    X <- tryCatch(model.matrix(form, data = coded_data[, -1])[,-1],
                  error = function(e) return(NULL))
    if (is.null(X)) return(list(OFS = FALSE, mean_corr = NA))
    blk_levels <- unique(as.character(coded_data$Block))
    blk_factor <- factor(as.character(coded_data$Block), levels = blk_levels)
    B <- matrix(0, nrow = nrow(coded_data), ncol = length(blk_levels))
    for (i in seq_len(nrow(coded_data))) {
      B[i, as.integer(blk_factor[i])] <- 1
    }
    colnames(B) <- paste0("Block", seq_len(ncol(B)))
    rownames(B) <- rownames(coded_data)
    Q <- diag(nrow(X)) - B %*% solve(t(B) %*% B) %*% t(B)
    X_adj <- Q %*% X
    Cmat <- t(X_adj) %*% X_adj + diag(1e-12, ncol(X_adj))
    d <- sqrt(diag(Cmat))
    d[d == 0 | is.na(d)] <- 1
    corC <- Cmat / (d %o% d)
    offdiag_mean <- mean(abs(corC[upper.tri(corC)]), na.rm = TRUE)
    list(OFS = (offdiag_mean < 1e-8), mean_corr = offdiag_mean)
  }

  compute_Hd_DD_mixed <- function(Td, q_vec, a = 1, b = 0.5) {
    if (!is.matrix(Td)) Td <- as.matrix(Td)
    n <- nrow(Td); s <- ncol(Td)
    hamming_col <- function(x) outer(x, x, Vectorize(function(i, j) as.numeric(i != j)))
    H_list <- lapply(1:s, function(k) hamming_col(Td[, k]))
    Hd_total <- Reduce("+", H_list)
    term1 <- -prod(a + (q_vec - 1) * b / q_vec)
    term2_sum <- 0
    for (i in 1:n) for (j in 1:n) {
      prod_ij <- prod(sapply(1:s, function(k) a * (b / a)^(H_list[[k]][i, j])))
      term2_sum <- term2_sum + prod_ij
    }
    term2 <- term2_sum / (n^2)
    DD2 <- term1 + term2
    DD  <- sqrt(abs(DD2))
    list(Hd_total = Hd_total, DD = DD)
  }

  ## ---- helpers with verbose argument ----

  compute_Bvec_mixed <- function(Td, q_vec, verbose = FALSE) {
    if (!is.matrix(Td)) Td <- as.matrix(Td)
    n <- nrow(Td); s <- ncol(Td)
    if (length(q_vec) != s)
      stop("Length of q_vec must equal number of factors (columns in Td).")

    Bvec <- numeric(s)

    for (m in 1:s) {
      combs <- combn(s, m)
      Blist <- numeric(ncol(combs))

      for (j in seq_len(ncol(combs))) {
        cols <- combs[, j]
        q_sub <- q_vec[cols]
        q_prod <- prod(q_sub)
        idx <- apply(Td[, cols, drop = FALSE], 1, function(r) {
          sum((r - 1) * cumprod(c(1, head(q_sub, -1)))) + 1
        })
        counts <- tabulate(idx, nbins = q_prod)
        Blist[j] <- sum((counts - n / q_prod)^2)
      }
      Bvec[m] <- mean(Blist)
    }

    if (verbose) {
      cat("\n================= B-Criterion Results =================\n")
      cat("Levels per factor:", q_vec, "\n")
      cat("Balance vector (B1, B2, ..., Bs):\n")
      print(round(Bvec, 6))

      if (all(Bvec == 0)) {
        cat("The design is a perfect orthogonal array (balanced at all levels).\n")
      } else {
        cat("Nonzero values indicate deviation from orthogonality:\n")
        cat("   Smaller B_m means closer to orthogonal of strength m.\n")
      }
    }

    Bvec
  }

  compute_Es2_mixed_with_OFS <- function(Td, verbose = FALSE) {
    if (!is.matrix(Td)) Td <- as.matrix(Td)
    n <- nrow(Td); s <- ncol(Td)
    q_vec <- apply(Td, 2, function(x) length(unique(x)))
    X <- Td
    for (j in 1:s) {
      q <- q_vec[j]
      if (q %% 2 == 1) {
        coded <- seq(-(q - 1) / 2, (q - 1) / 2, length.out = q)
      } else {
        half <- q / 2
        coded <- seq(-half, half, length.out = q + 1)
        coded <- coded[coded != 0]
      }
      X[, j] <- coded[Td[, j]]
    }
    S <- crossprod(X)
    sumsq <- 0
    for (i in 1:(s - 1)) {
      for (j in (i + 1):s) {
        sumsq <- sumsq + S[i, j]^2
      }
    }
    E_s2 <- sumsq / choose(s, 2)

    if (verbose) {
      cat("\n================= E_s^2 =================\n")
      cat("Runs (n):", n, "\t Factors (s):", s, "\n")
      cat("Levels per factor:", q_vec, "\n")
      cat("\nCross-product matrix (S = X'X):\n")
      print(round(S, 3))
      cat("\nSum of squared off-diagonals =", round(sumsq, 6), "\n")
      cat("E_s^2 =", round(E_s2, 6), "\n")
    }

    invisible(list(S = S, E_s2 = E_s2))
  }

  ## ===== Das et al. helpers (unchanged except for message() in error) ===== ##
  if (!exists("das_ginv_safe", mode = "function")) {
    das_ginv_safe <- function(A, tol = 1e-8) {
      return(MASS::ginv(as.matrix(A), tol = tol))
    }
  }

  if (!exists("das_compute_Cmatrix", mode = "function")) {
    das_compute_Cmatrix <- function(N, k_vec = NULL, r_vec = NULL) {
      N <- as.matrix(N)
      v <- nrow(N); b <- ncol(N)
      if (is.null(k_vec)) k_vec <- colSums(N)
      if (is.null(r_vec)) r_vec <- rowSums(N)
      if (length(k_vec) != b) stop("k_vec must have length equal to number of blocks (ncol(N))")
      if (length(r_vec) != v) stop("r_vec must have length equal to number of treatments (nrow(N))")
      if (any(k_vec == 0)) stop("Some block sizes are zero")
      Rmat <- diag(as.numeric(r_vec), nrow = v)
      Kinv <- diag(1 / as.numeric(k_vec), nrow = b)
      C <- Rmat - N %*% Kinv %*% t(N)
      return(list(C = as.matrix(C), R = Rmat, Kinv = Kinv, N = N, r_vec = r_vec, k_vec = k_vec))
    }
  }

  if (!exists("das_make_Pi", mode = "function")) {
    das_make_Pi <- function(s) {
      s <- as.integer(s)
      if (s <= 1) stop("Each factor must have at least 2 levels for contrasts to be meaningful.")
      H <- contr.helmert(s)
      first_col <- rep(1 / sqrt(s), s)
      Q <- qr.Q(qr(H))
      P <- cbind(first_col, Q)
      if (ncol(P) < s) {
        last_extra <- qr.Q(qr(matrix(rnorm(s * (s - ncol(P))), nrow = s)))
        needed <- s - ncol(P)
        P <- cbind(P, last_extra[, 1:needed, drop = FALSE])
      }
      if (sum(P[, 1]) < 0) P[, 1] <- -P[, 1]
      return(P)
    }
  }

  if (!exists("das_build_effect_contrasts", mode = "function")) {
    das_build_effect_contrasts <- function(s_vec, alpha) {
      m <- length(s_vec)
      if (length(alpha) != m) stop("alpha length must equal number of factors")
      Pi_list <- lapply(s_vec, das_make_Pi)
      col_blocks <- lapply(seq_len(m), function(i) {
        Pi <- Pi_list[[i]]
        if (alpha[i] == 0) {
          return(Pi[, 1, drop = FALSE])
        } else {
          return(Pi[, -1, drop = FALSE])
        }
      })
      Tcon <- col_blocks[[1]]
      if (m >= 2) {
        for (i in 2:m) {
          Tcon <- kronecker(Tcon, col_blocks[[i]])
        }
      }
      return(Tcon)
    }
  }

  if (!exists("das_enumerate_all_effects", mode = "function")) {
    das_enumerate_all_effects <- function(s_vec) {
      m <- length(s_vec)
      all_alpha <- do.call(expand.grid, rep(list(c(0, 1)), m))
      res <- list()
      for (i in seq_len(nrow(all_alpha))) {
        alpha <- as.integer(all_alpha[i, ])
        if (all(alpha == 0)) next
        Tcon <- das_build_effect_contrasts(s_vec, alpha)
        res[[length(res) + 1]] <- list(alpha = alpha, contrasts = Tcon)
      }
      return(res)
    }
  }

  if (!exists("das_check_effect_balance", mode = "function")) {
    das_check_effect_balance <- function(Cmat, contrasts, tol = 1e-6) {
      v <- nrow(Cmat)
      Cg <- tryCatch(das_ginv_safe(Cmat), error = function(e) NULL)
      if (is.null(Cg)) stop("Failed to invert or pseudo-invert C")
      cov_block <- t(contrasts) %*% Cg %*% contrasts
      estimable_cols <- apply(contrasts, 2,
                              function(col) sqrt(sum((Cmat %*% col)^2)) > tol)
      estimable <- all(estimable_cols)
      f <- ncol(cov_block)
      s_est <- sum(diag(cov_block)) / f
      residual_mat <- cov_block - s_est * diag(f)
      balanced <- max(abs(residual_mat)) < tol
      return(list(estimable = estimable, balanced = balanced, scalar = s_est,
                  cov = cov_block, max_offdiag = max(abs(residual_mat))))
    }
  }

  if (!exists("das_check_pairwise_orthogonality", mode = "function")) {
    das_check_pairwise_orthogonality <- function(Cmat, T1, T2, tol = 1e-6) {
      Cg <- das_ginv_safe(Cmat)
      crosscov <- t(T1) %*% Cg %*% T2
      return(list(is_orthogonal = max(abs(crosscov)) < tol,
                  crosscov = crosscov,
                  max_abs = max(abs(crosscov))))
    }
  }

  if (!exists("das_pencilwise_grouping", mode = "function")) {
    das_pencilwise_grouping <- function(s_vec) {
      if (!all(s_vec == s_vec[1])) {
        warning("Pencilwise grouping implemented only for symmetric designs (all s_i identical). Returning simple groups by order of interaction.")
      }
      m <- length(s_vec)
      all_alpha <- do.call(expand.grid, rep(list(c(0, 1)), m))
      res <- split(as.data.frame(all_alpha)[-1, , drop = FALSE],
                   rowSums(as.matrix(all_alpha[-1, ])))
      return(res)
    }
  }

  if (!exists("das_analyze_design", mode = "function")) {
    das_analyze_design <- function(s_vec, N, check_all_pairs = FALSE, tol = 1e-6) {
      cm <- das_compute_Cmatrix(N)
      C  <- cm$C
      v  <- nrow(C)
      eigs <- eigen((C + t(C)) / 2, symmetric = TRUE, only.values = TRUE)$values
      rankC <- qr(C)$rank
      effects <- das_enumerate_all_effects(s_vec)
      effects_summary <- list()
      for (i in seq_along(effects)) {
        alpha <- effects[[i]]$alpha
        Tcon  <- effects[[i]]$contrasts
        chk   <- das_check_effect_balance(C, Tcon, tol = tol)
        effects_summary[[i]] <- list(alpha = alpha,
                                     f_alpha = ncol(Tcon),
                                     estimable = chk$estimable,
                                     balanced = chk$balanced,
                                     balance_scalar = chk$scalar,
                                     max_offdiag = chk$max_offdiag)
      }
      pairwise <- NULL
      if (check_all_pairs) {
        pairwise <- list()
        idx <- 1
        for (i in seq_along(effects)) {
          for (j in seq_along(effects)) {
            if (j <= i) next
            T1 <- effects[[i]]$contrasts
            T2 <- effects[[j]]$contrasts
            orth <- das_check_pairwise_orthogonality(C, T1, T2, tol = tol)
            pairwise[[idx]] <- list(alpha1 = effects[[i]]$alpha,
                                    alpha2 = effects[[j]]$alpha,
                                    is_orthogonal = orth$is_orthogonal,
                                    max_abs = orth$max_abs)
            idx <- idx + 1
          }
        }
      }
      return(list(C = C, R = cm$R, Kinv = cm$Kinv, N = cm$N, eig = eigs,
                  rank = rankC, effects = effects_summary,
                  pairwise = pairwise))
    }
  }

  compute_Kt <- function(D, max_t = 4, weights = NULL, verbose = FALSE) {
    if (!is.matrix(D)) D <- as.matrix(D)
    n <- nrow(D); s <- ncol(D)
    if (is.null(weights)) weights <- rep(1, s)

    X <- D
    for (j in 1:s) {
      levs <- sort(unique(D[, j]))
      X[, j] <- seq(-1, 1, length.out = length(levs))[match(D[, j], levs)]
    }

    K <- numeric(max_t)
    for (t in 1:max_t) {
      K[t] <- mean((X %*% weights / sum(weights))^t)
    }

    if (verbose) {
      cat("\n================= MMA (Moment Matrix Analysis) =================\n")
      cat("K-vector (moments 1..", max_t, "): ", paste(round(K, 6), collapse = ", "), "\n")
    }

    invisible(list(K = K))
  }

  compute_Aj_twolevel <- function(D, max_j = 3, verbose = FALSE) {
    if (!is.matrix(D)) D <- as.matrix(D)
    n <- nrow(D); s <- ncol(D)
    max_j <- min(max_j, s)
    X <- 2 * D - 1
    A <- numeric(max_j)
    for (j in 1:max_j) {
      combs <- combn(s, j)
      vals <- numeric(ncol(combs))
      for (k in seq_len(ncol(combs))) {
        prod_mat <- apply(X[, combs[, k], drop = FALSE], 1, prod)
        vals[k] <- mean(prod_mat)^2
      }
      A[j] <- mean(vals)
    }

    if (verbose) {
      cat("\n================= MA (2-level design) =================\n")
      cat("A-vector (A1..A", max_j, "): ", paste(round(A, 6), collapse = ", "), "\n")
    }

    invisible(list(A = A))
  }

  compute_Aj_mixed <- function(D, max_j = 3, verbose = FALSE) {
    if (!is.matrix(D)) D <- as.matrix(D)
    n <- nrow(D); s <- ncol(D)
    max_j <- min(max_j, s)
    X <- scale(D, center = TRUE, scale = TRUE)
    A <- numeric(max_j)
    for (j in 1:max_j) {
      combs <- combn(s, j)
      vals <- numeric(ncol(combs))
      for (k in seq_len(ncol(combs))) {
        prod_mat <- apply(X[, combs[, k], drop = FALSE], 1, prod)
        vals[k] <- mean(prod_mat^2)
      }
      A[j] <- mean(vals)
    }

    if (verbose) {
      cat("\n================= GMA (Mixed-level design) =================\n")
      cat("A-vector (A1..A", max_j, "): ", paste(round(A, 6), collapse = ", "), "\n")
    }

    invisible(list(A = A))
  }

  decode_treatment <- function(trt_label) {
    t(sapply(trt_label, function(x) as.numeric(unlist(strsplit(x, "")))))
  }

  ## ---- main computations ----

  built <- build_incidence(factor_levels, blocks)
  comp  <- compute_Cmatrix(built$N)

  Cmat_full <- comp$C
  r_vec <- comp$r
  v <- nrow(Cmat_full)

  Cplus <- tryCatch(mpinv2(Cmat_full), error = function(e) NULL)

  if (length(unique(round(r_vec, 12))) == 1) {
    r_scalar <- as.numeric(r_vec[1])
    if (r_scalar <= 0) {
      Fmat <- matrix(0, v, v)
    } else {
      Fmat <- Cmat_full / r_scalar
    }
    equirep_flag <- TRUE
  } else {
    invR <- diag(1 / pmax(r_vec, .Machine$double.eps))
    Fmat <- invR %*% Cmat_full
    equirep_flag <- FALSE
  }

  conn_flag <- (as.integer(Matrix::rankMatrix(Cmat_full)[1]) == (v - 1))

  if (!conn_flag) {
    A_eff <- 0; D_eff <- 0; E_eff <- 0; MV_eff <- 0
  } else {
    ev <- eigen(Fmat, symmetric = TRUE, only.values = TRUE)$values
    tol_eig <- 1e-8
    deltas <- ev[ev > tol_eig]
    if (length(deltas) < (v - 1)) {
      deltas <- sort(deltas, decreasing = TRUE)
    }
    deltas <- sort(deltas, decreasing = FALSE)
    if (length(deltas) > (v - 1)) deltas <- tail(deltas, v - 1)

    inv_sum <- sum(1 / pmax(deltas, tol_eig))
    A_eff <- if (inv_sum > 0) (v - 1) / inv_sum else 0
    D_eff <- if (all(deltas > 0)) prod(deltas)^(1 / (v - 1)) else 0
    E_eff <- min(deltas)

    if (!is.null(Cplus) && equirep_flag && r_scalar > 0) {
      pairs <- combn(v, 2)
      e_ij_vals <- numeric(ncol(pairs))
      for (idx in seq_len(ncol(pairs))) {
        i <- pairs[1, idx]; j <- pairs[2, idx]
        l <- rep(0, v); l[i] <- 1; l[j] <- -1
        denom <- as.numeric(t(l) %*% Cplus %*% l)
        if (denom <= 0) e_ij_vals[idx] <- 0
        else           e_ij_vals[idx] <- 2 / (r_scalar * denom)
      }
      MV_eff <- min(e_ij_vals, na.rm = TRUE)
    } else {
      MV_eff <- NA_real_
    }
  }

  mu_vals <- eigen(Cmat_full, symmetric = TRUE, only.values = TRUE)$values
  tol_eig <- 1e-8
  mu_nonzero <- mu_vals[mu_vals > tol_eig]
  deltas_mu <- sort(mu_nonzero / ifelse(equirep_flag, r_scalar, 1),
                    decreasing = FALSE)

  p_values <- c(2, 1, 0.5)
  phi_p_values <- sapply(p_values, function(p) {
    if (any(deltas_mu <= 0)) return(Inf)
    mean(deltas_mu^(-p))
  })
  names(phi_p_values) <- paste0("phi_p (p=", p_values, ")")

  trC2 <- sum(mu_nonzero^2)
  equal_val <- mean(mu_nonzero)
  euclid_dist <- sqrt(sum((mu_nonzero - equal_val)^2))

  is_completely_symmetric <- function(C, tol = 1e-8) {
    diag_vals <- diag(C)
    off_vals  <- C[upper.tri(C)]
    if (max(abs(diag_vals - mean(diag_vals))) > tol) return(FALSE)
    if (max(abs(off_vals  - mean(off_vals)))  > tol) return(FALSE)
    TRUE
  }
  sym_flag <- is_completely_symmetric(Cmat_full)
  uni_msg <- if (sym_flag)
    "Design meets symmetry condition -> candidate universally optimal"
  else
    "Symmetry condition not met -> unlikely universally optimal by Kiefer's sufficient condition"

  compute_J2_distance <- function(D, w = NULL) {
    D <- as.matrix(D); n <- nrow(D); m <- ncol(D)
    if (is.null(w)) w <- rep(1, m)
    delta_sq_sum <- 0
    for (i in 1:(n - 1)) {
      for (j in (i + 1):n) {
        delta_ij <- sum(w * (D[i, ] == D[j, ]))
        delta_sq_sum <- delta_sq_sum + delta_ij^2
      }
    }
    BalanceCoeff <- delta_sq_sum / choose(n, 2)
    list(J2_Dist = delta_sq_sum, BalanceCoeff = BalanceCoeff)
  }

  compute_J2_optimization <- function(D, w = NULL) {
    D <- as.matrix(D); n <- nrow(D); m <- ncol(D)
    if (is.null(w)) w <- rep(1, m)
    delta_sq_sum <- 0
    for (i in 1:(n - 1)) {
      for (j in (i + 1):n) {
        delta_ij <- sum(w * (D[i, ] == D[j, ]))
        delta_sq_sum <- delta_sq_sum + delta_ij^2
      }
    }
    BalanceCoeff <- sum(apply(D, 2, function(col) {
      freq <- table(col) / n
      sum((freq - mean(freq))^2)
    }))
    list(J2_Opt = delta_sq_sum, BalanceCoeff = BalanceCoeff)
  }

  compute_J2_lower_bound <- function(D, w = NULL, s_vec = NULL) {
    D <- as.matrix(D)
    n <- nrow(D); m <- ncol(D)
    if (is.null(w)) w <- rep(1, m)
    if (is.null(s_vec)) s_vec <- apply(D, 2, function(col) length(unique(col)))
    L_sum <- 0
    for (k in 1:m) {
      Sk <- s_vec[k]; wk <- w[k]
      nk <- n / Sk
      L_sum <- L_sum + wk^2 * (nk^2 * Sk + nk * (nk - 1) * Sk)
    }
    0.5 * L_sum
  }

  factor_matrix_runs <- do.call(rbind, lapply(blocks, function(bl) decode_treatment(bl)))
  J2_Dist_res <- compute_J2_distance(factor_matrix_runs)
  J2_Opt_res  <- compute_J2_optimization(factor_matrix_runs)
  J2_LB_val   <- compute_J2_lower_bound(factor_matrix_runs,
                                        w = rep(1, ncol(factor_matrix_runs)),
                                        s_vec = factor_levels)

  J2_eff_dist <- if (!is.null(J2_Dist_res$J2_Dist) &&
                     J2_Dist_res$J2_Dist > 0)
    J2_LB_val / J2_Dist_res$J2_Dist else NA
  J2_eff_opt  <- if (!is.null(J2_Opt_res$J2_Opt) &&
                     J2_Opt_res$J2_Opt > 0)
    J2_LB_val / J2_Opt_res$J2_Opt else NA

  mm      <- build_contrast_matrix(built$grids)
  proper  <- (max(colSums(built$N)) - min(colSums(built$N))) < 1e-8
  equi    <- (max(rowSums(built$N)) - min(rowSums(built$N))) < 1e-8
  conn    <- (as.integer(Matrix::rankMatrix(comp$C)[1]) == (nrow(comp$C) - 1))
  balance <- evaluate_balance(comp$C, mm)
  ofs     <- check_design_OFS(factor_levels, blocks)

  results_list <- lapply(blocks, function(b) {
    Td_block <- decode_treatment(b)
    compute_Hd_DD_mixed(Td_block, q_vec = factor_levels)
  })
  Td_all <- do.call(rbind, lapply(blocks, decode_treatment))
  overall_result <- compute_Hd_DD_mixed(Td_all, q_vec = factor_levels)

  ## ------------- PRINTING SECTION (all wrapped in verbose) -------------
  if (verbose) {
    cat("\n====================== DESIGN PROPERTIES ======================\n")
    cat("Factorial Experiment:", paste(factor_levels, collapse = " x "), "\n")
    cat("Number of Treatments (v):", nrow(built$N), "\n")
    cat("Number of Blocks (b):", ncol(built$N), "\n")
    cat("Proper Design:", proper, "\n")
    cat("Equireplicate:", equi, "\n")
    cat("Connected:", conn, "\n\n")
    cat("OFS (Orthogonal Factorial Structure):", ofs$OFS,
        "(mean |correlation| =", round(ofs$mean_corr, 3), ")\n")
  }

  s_vec <- factor_levels
  Nmat  <- built$N

  das_res <- NULL
  if (ncol(Nmat) == 0 || nrow(Nmat) == 0) {
    if (verbose) cat("Skipping Das diagnostics: empty incidence matrix.\n")
  } else {
    das_res <- tryCatch({
      das_analyze_design(s_vec, Nmat, check_all_pairs = FALSE, tol = 1e-6)
    }, error = function(e) {
      if (verbose) message("Das diagnostics error: ", e$message)
      NULL
    })

    if (!is.null(das_res) && verbose) {
      cat("rank(C) =", das_res$rank, "\n")
      cat("Eigenvalues of C:\n")
      print(round(das_res$eig, 6))

      cat("\n================ EFFECTS SUMMARY (All Effects) ================\n")
      cat(sprintf("%-10s %-10s %-10s %-12s %-10s %-15s %-15s\n",
                  "alpha", "DF", "estimable", "balanced",
                  "balance_variance", "max_offdiag(OFS)", " "))

      for (ef in das_res$effects) {
        cat(sprintf("%-10s %-10d %-10s %-12s %-10.6g %-15.3g\n",
                    paste(ef$alpha, collapse = ""),
                    ef$f_alpha,
                    ef$estimable,
                    ef$balanced,
                    ef$balance_scalar,
                    ef$max_offdiag))
      }
      cat("=================================================================\n")
    }
  }

  Td_all_1based <- decode_treatment(unlist(blocks)) + 1

  if (verbose) cat("\n-> Overall Design:\n")
  K_res <- compute_Kt(Td_all_1based, max_t = 4, verbose = verbose)

  if (all(apply(Td_all_1based, 2, function(x) length(unique(x))) == 2)) {
    A_res <- compute_Aj_twolevel(Td_all_1based, max_j = 3, verbose = verbose)
  } else {
    A_res <- compute_Aj_mixed(Td_all_1based, max_j = 3, verbose = verbose)
  }

  if (verbose) {
    cat("\nHamming Distance Matrix of Combined Design:\n")
    print(overall_result$Hd_total)

    cat("\n=========================== Discrete Discrepancy Summary ===========================\n")
  }
  block_DDs <- sapply(results_list, function(x) x$DD)
  if (verbose) {
    cat("Average DD =", round(mean(block_DDs), 6), "\n")
    cat("Overall DD =", round(overall_result$DD, 6), "\n")
  }

  Bvec_overall <- compute_Bvec_mixed(Td_all_1based, q_vec = factor_levels,
                                     verbose = verbose)

  if (verbose) cat("\n-> Overall Design:\n")
  Es2_res <- compute_Es2_mixed_with_OFS(Td_all_1based, verbose = verbose)

  if (verbose) {
    cat("\n======================= EFFICIENCY MEASURES ======================\n")
    cat(sprintf("A-efficiency = %g\n", A_eff))
    cat(sprintf("D-efficiency = %g\n", D_eff))
    cat(sprintf("E-efficiency = %g\n", E_eff))
    if (is.na(MV_eff))
      cat("MV-efficiency = NA (unequal replication: MV from pairwise contrasts not computed)\n")
    else
      cat(sprintf("MV-efficiency = %g\n", MV_eff))

    cat("\n===================== ADVANCED OPTIMALITY & ABERRATION =====================\n")
    cat(sprintf("J2 (Distance Function): %g   | Balance Coeff (Form II): %g\n",
                J2_Dist_res$J2_Dist, J2_Dist_res$BalanceCoeff))
    cat(sprintf("   -> J2-efficiency (L(n)/J2) = %g\n", J2_eff_dist))
    cat(sprintf("J2 (Optimization Procedure): %g   | Balance Coeff (Form I): %g\n",
                J2_Opt_res$J2_Opt, J2_Opt_res$BalanceCoeff))
    cat(sprintf("   -> J2-efficiency (L(n)/J2) = %g\n", J2_eff_opt))
    cat(sprintf("Lower Bound of J2 (Balanced Design): %g\n", J2_LB_val))
    cat("Phi_p values:\n"); print(phi_p_values)
    cat("\nS-criterion: trace(C^2) = ", trC2, "\n")
    cat("Distance from equal eigenvalues = ", euclid_dist, "\n")
    cat("\nUniversal optimality check: ", uni_msg, "\n")
  }

  ## ----------------- OBJECT TO RETURN -----------------
  out <- list(
    factor_levels   = factor_levels,
    blocks          = blocks,
    incidence       = built$N,
    treatments      = built$treatments,
    grids           = built$grids,
    C               = Cmat_full,
    r               = r_vec,
    k               = comp$k,
    proper          = proper,
    equireplicate   = equi,
    connected       = conn,
    ofs             = ofs,
    balance         = balance,
    das             = das_res,
    efficiencies    = list(A = A_eff, D = D_eff, E = E_eff, MV = MV_eff),
    J2              = list(
      distance      = J2_Dist_res,
      optimization  = J2_Opt_res,
      lower_bound   = J2_LB_val,
      eff_distance  = J2_eff_dist,
      eff_optimization = J2_eff_opt
    ),
    phi_p           = phi_p_values,
    traceC2         = trC2,
    eigen_equal_dist = euclid_dist,
    symmetry        = list(is_symmetric = sym_flag, message = uni_msg),
    Hd_total        = overall_result$Hd_total,
    DD              = list(block = block_DDs, overall = overall_result$DD),
    Bvec            = Bvec_overall,
    Es2             = Es2_res,
    MMA             = K_res,
    A_vector        = A_res
  )

  invisible(out)
}
