% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network.regularization.R
\name{network.regularization}
\alias{network.regularization}
\title{Regularized Networks with Convex and Non-convex Penalties}
\usage{
network.regularization(
  data,
  n = NULL,
  corr = c("auto", "cor_auto", "cosine", "pearson", "spearman"),
  na.data = c("pairwise", "listwise"),
  penalty = c("atan", "bridge", "l1", "l2", "lomax", "mcp", "scad", "weibull"),
  gamma = NULL,
  lambda = NULL,
  nlambda = 50,
  lambda.min.ratio = 0.01,
  penalize.diagonal = TRUE,
  optimize.lambda = FALSE,
  ic = c("AIC", "AICc", "BIC", "EBIC"),
  ebic.gamma = 0.5,
  fast = TRUE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis}

\item{n}{Numeric (length = 1).
Sample size \strong{must} be provided if \code{data} provided is a correlation matrix}

\item{corr}{Character (length = 1).
Method to compute correlations.
Defaults to \code{"auto"}.
Available options:

\itemize{

\item \code{"auto"} --- Automatically computes appropriate correlations for
the data using Pearson's for continuous, polychoric for ordinal,
tetrachoric for binary, and polyserial/biserial for ordinal/binary with
continuous. To change the number of categories that are considered
ordinal, use \code{ordinal.categories}
(see \code{\link[EGAnet]{polychoric.matrix}} for more details)

\item \code{"cor_auto"} --- Uses \code{\link[qgraph]{cor_auto}} to compute correlations.
Arguments can be passed along to the function

\item \code{"cosine"} --- Uses \code{\link[EGAnet]{cosine}} to compute cosine similarity

\item \code{"pearson"} --- Pearson's correlation is computed for all
variables regardless of categories

\item \code{"spearman"} --- Spearman's rank-order correlation is computed
for all variables regardless of categories

}

For other similarity measures, compute them first and input them
into \code{data} with the sample size (\code{n})}

\item{na.data}{Character (length = 1).
How should missing data be handled?
Defaults to \code{"pairwise"}.
Available options:

\itemize{

\item \code{"pairwise"} --- Computes correlation for all available cases between
two variables

\item \code{"listwise"} --- Computes correlation for all complete cases in the dataset

}}

\item{penalty}{Character (length = 1).
Available options:

\itemize{

\item \code{"atan"} --- Arctangent (Wang & Zhu, 2016)
\deqn{\lambda \cdot (\gamma + 2 \pi) \cdot \arctan(\frac{|x|}{\gamma})}

\item \code{"bridge"} --- Bridge (Fu, 1998)
\deqn{\lambda \cdot |x|^\gamma}

\item \code{"l1"} --- LASSO (Tibshirani, 1996)
\deqn{\lambda \cdot |x|}

\item \code{"l2"} --- Ridge (Hoerl & Kennard, 1970)
\deqn{\lambda \cdot x^2}

\item \code{"lomax"} --- Lomax (Lomax, 1951)
\deqn{\lambda \cdot (1 - (\frac{1}{(|x| + 1)^\gamma}))}

\item \code{"mcp"} --- Minimax Concave Penalty (Zhang, 2010)
\deqn{
P(x; \lambda, \gamma) =
\begin{cases}
\lambda |x| - \frac{x^2}{2\gamma} & \text{if } |x| \leq \gamma\lambda \\
\frac{\gamma \lambda^2}{2} & \text{if } |x| > \gamma\lambda
\end{cases}
}

\item \code{"scad"} --- Smoothly Clipped Absolute Deviation (Fan & Li, 2001)
\deqn{
P(x; \lambda, \gamma) =
\begin{cases}
\lambda |x| & \text{if } |x| \leq \lambda \\
-\frac{|x|^2 - 2\gamma\lambda|x| + \lambda^2}{2(\gamma - 1)} & \text{if } \lambda < |x| \leq \gamma\lambda \\
\frac{(\gamma + 1)\lambda^2}{2} & \text{if } |x| > \gamma\lambda
\end{cases}
}

}}

\item{gamma}{Numeric (length = 1).
Adjusts the shape of the penalty.
Defaults:

\itemize{

\item \code{"atan"} = 0.01

\item \code{"bridge"} = 1

\item \code{"lomax"} = 4

\item \code{"mcp"} = 3

\item \code{"scad"} = 3.7

}}

\item{lambda}{Numeric (length = 1).
Adjusts the initial penalty provided to the non-convex penalty function}

\item{nlambda}{Numeric (length = 1).
Number of lambda values to test.
Defaults to \code{100}}

\item{lambda.min.ratio}{Numeric (length = 1).
Ratio of lowest lambda value compared to maximal lambda.
Defaults to \code{0.01}}

\item{penalize.diagonal}{Boolean (length = 1).
Should the diagonal be penalized?
Defaults to \code{FALSE}}

\item{optimize.lambda}{Boolean (length = 1).
Whether optimization of lambda should be performed.
Defaults to \code{FALSE} or grid search over lambda.
If \code{TRUE}, then \code{\link[stats]{optimize}} is used
to find the optimal lambda}

\item{ic}{Character (length = 1).
What information criterion should be used for model selection?
Available options include:

\itemize{

\item \code{"AIC"} --- Akaike's information criterion: \eqn{-2L + 2E}

\item \code{"AICc"} --- AIC corrected: \eqn{AIC + \frac{2E^2 + 2E}{n - E - 1}}

\item \code{"BIC"} --- Bayesian information criterion: \eqn{-2L + E \cdot \log{(n)}}

\item \code{"EBIC"} --- Extended BIC: \eqn{BIC + 4E \cdot \gamma \cdot \log{(E)}}

}

Term definitions:

\itemize{

\item \eqn{n} --- sample size

\item \eqn{p} --- number of variables

\item \eqn{E} --- edges

\item \eqn{S} --- empirical correlation matrix

\item \eqn{K} --- estimated inverse covariance matrix (network)

\item \eqn{L = \frac{n}{2} \cdot \log \text{det} K - \sum_{i=1}^p (SK)_{ii}}

}

Defaults to \code{"BIC"}}

\item{ebic.gamma}{Numeric (length = 1)
Value to set gamma parameter in EBIC (see above).
Defaults to \code{0.50}

\emph{Only used if \code{ic = "EBIC"}}}

\item{fast}{Boolean (length = 1).
Whether the \code{\link[glassoFast]{glassoFast}} version should be used
to estimate the GLASSO.
Defaults to \code{TRUE}.

The fast results \emph{may} differ by less than floating point of the original
GLASSO implemented by \code{\link[glasso]{glasso}} and should not impact reproducibility much (set to \code{FALSE} if concerned)}

\item{verbose}{Boolean (length = 1).
Whether messages and (insignificant) warnings should be output.
Defaults to \code{FALSE} (silent calls).
Set to \code{TRUE} to see all messages and warnings for every function call}

\item{...}{Additional arguments to be passed on to \code{\link[EGAnet]{auto.correlate}}}
}
\value{
A network matrix
}
\description{
A general function to estimate Gaussian graphical models using
regularization penalties. All non-convex penalties are implemented using
the Local Linear Approximation (LLA: Fan & Li, 2001)
}
\examples{
# Obtain data
wmt <- wmt2[,7:24]

# Obtain network
l1_network <- network.regularization(data = wmt)

}
\references{
\strong{SCAD penalty and Local Linear Approximation} \cr
Fan, J., & Li, R. (2001).
Variable selection via nonconcave penalized likelihood and its oracle properties.
\emph{Journal of the American Statistical Association}, \emph{96}(456), 1348--1360.

\strong{Bridge penalty} \cr
Fu, W. J. (1998). Penalized regressions: The bridge versus the lasso.
\emph{Journal of Computational and Graphical Statistics}, \emph{7}(3), 397--416.

\strong{L2 penalty} \cr
Hoerl, A. E., & Kennard, R. W. (1970).
Ridge regression: Biased estimation for nonorthogonal problems.
\emph{Technometrics}, \emph{12}(1), 55--67.

\strong{Lomax penalty} \cr
Lomax, K. S. (1954).
Business failures: Another example of the analysis of failure data.
\emph{Journal of the American Statistical Association}, \emph{49}(268), 847--852.

\strong{L1 penalty} \cr
Tibshirani, R. (1996).
Regression shrinkage and selection via the lasso.
\emph{Journal of the Royal Statistical Society: Series B (Methodological)}, \emph{58}(1), 267--288.

\strong{Atan penalty} \cr
Wang, Y., & Zhu, L. (2016).
Variable selection and parameter estimation with the Atan regularization method.
\emph{Journal of Probability and Statistics}, \emph{2016}, 1--12.

\strong{Original simulation in psychometric networks} \cr
Williams, D. R. (2020).
Beyond lasso: A survey of nonconvex regularization in Gaussian graphical models.
\emph{PsyArXiv}.

\strong{MCP penalty} \cr
Zhang, C.-H. (2010).
Nearly unbiased variable selection under minimax concave penalty.
\emph{Annals of Statistics}, \emph{38}(2), 894--942.
}
\author{
Alexander P. Christensen <alexpaulchristensen at gmail.com> and
Hudson Golino <hfg9s at virginia.edu>
}
