% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/process_ngs.R
\name{process_ngs}
\alias{process_ngs}
\title{Process next generation sequencing data}
\usage{
process_ngs(
  X,
  sample_data,
  taxa_table = NULL,
  phylo_tree = NULL,
  remove_ids = NULL,
  min_reads = 500,
  min_prev = 0.1,
  normalise = c("load", "TSS", "none"),
  load_colname = NULL,
  min_load = 10000,
  transform = c("clr", "log", "none"),
  impute_control = list(method = "GBM", output = "p-counts", z.delete = FALSE, z.warning
    = 1, suppress.print = TRUE),
  raw_phyloseq = TRUE,
  eco_phyloseq = TRUE,
  return_all = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{X}{A numeric matrix or data frame of raw counts with samples as rows
and features (e.g., taxa) as columns. Row names must be sample IDs.}

\item{sample_data}{A data frame containing sample-level data.
Must include a column named \code{sample_id} with matching row names with \code{X}.}

\item{taxa_table}{Optional. Taxonomy annotation table to build \code{phyloseq} objects.
Row names must match column names of \code{X}.}

\item{phylo_tree}{Optional. Phylogenetic tree to add to \code{phyloseq} objects.}

\item{remove_ids}{A regex or character vector to filter rows in \code{X}. Set to \code{NULL} to skip.}

\item{min_reads}{Numeric. Minimum number of total reads required per sample.
Default is 500.}

\item{min_prev}{Numeric between 0 and 1. Minimum feature prevalence threshold.
Default is 0.1 (i.e., feature must be present in >= 10 \% of samples).}

\item{normalise}{Normalization method. One of \code{"load"} (microbial load data),
\code{"TSS"} (total sum scaling), or \code{"none"}.}

\item{load_colname}{Column name in \code{sample_data} containing microbial load values.
Required if \code{normalise = "load"}.}

\item{min_load}{Numeric. Default is 1e4. Warns if any microbial load value < min_load.}

\item{transform}{Transformation method. One of \code{"clr"} (centered log-ratio with zero imputation),
\code{"log"} (pseudo-log using \code{log1p()}), or \code{"none"}.
Note: When using \code{"clr"}, zero values are imputed using \code{zCompositions::cmultRepl()}.}

\item{impute_control}{A named list of arguments to be passed to \code{zCompositions::cmultRepl()}.}

\item{raw_phyloseq}{Logical. If \code{TRUE}, constructs a \code{phyloseq} object with
the table of raw counts (filtered failed runs if needed). Default is \code{TRUE}.}

\item{eco_phyloseq}{Logical. If \code{TRUE}, constructs a \code{phyloseq} object
with the ecosystem abundances (i.e. after \code{normalise = "load"}). Default is \code{TRUE}.}

\item{return_all}{Logical. If \code{TRUE}, additional intermediate data matrices
(\code{X_matched}, \code{X_norm}, \code{X_prev}) are included in the output. Default is \code{FALSE}.}

\item{verbose}{Logical. If \code{TRUE}, prints progress messages during execution.
Default is \code{TRUE}.}
}
\value{
A named list containing:
\describe{
\item{\code{X_processed}}{Matrix of processed feature counts after filtering,
normalization, and transformation.}
\item{\code{sdata_final}}{Matched and filtered \code{sample_data} corresponding to
retained samples.}
\item{\code{phyloseq_raw}}{\code{phyloseq} object created from raw filtered data.
\code{NULL} if \code{raw_phyloseq = FALSE}.}
\item{\code{phyloseq_eco}}{\code{phyloseq} object from ecosystem abundance data.
\code{NULL} if \code{eco_phyloseq = FALSE} or \code{normalise != "load"}.}
\item{\code{X_matched}}{(Optional) Matched and filtered count matrix, pre-normalization.
Returned only if \code{return_all = TRUE}.}
\item{\code{X_norm}}{(Optional) Normalized count matrix. Returned only if \code{return_all = TRUE}.}
\item{\code{X_prev}}{(Optional) Prevalence-filtered matrix, pre-transformation.
Returned only if \code{return_all = TRUE}.}
}
}
\description{
This function performs quality control, filtering, normalization, and transformation
of sequencing data raw counts.
It can also build \code{phyloseq} objects for downstream ecological analyses,
and optionally returns intermediate processing steps.
}
\details{
\itemize{
\item Zeros are imputed with \code{zCompositions::cmultRepl()} before CLR transformation.
\item QC or other samples are removed if \code{remove_ids} is specified.
\item Sample IDs in \code{X} and \code{sample_data} row names are matched and aligned.
\item Can generate both a \code{phyloseq_raw} phyloseq object containing raw counts and a
\code{phyloseq_eco} object with ecosystem counts, if a \code{load_colname} column from
\code{sample_data} is provided to normalize the counts by microbial load (recommended best practice).
}
}
\examples{
if (requireNamespace("phyloseq", quietly = TRUE)) {
mock_X <- matrix(sample(0:1000, 25, replace = TRUE),
                 nrow = 5,
                 dimnames = list(paste0("sample", 1:5),
                 paste0("ASV", 1:5))
                 )

mock_sample_data <- data.frame(
  sample_id = paste0("sample", 1:5),
  load = c(1e5, 2e5, 1e4, 5e4, 1.5e5),
  condition = factor(rep(c("A", "B"), length.out = 5)),
  row.names = paste0("sample", 1:5)
  )

mock_taxa_table <- data.frame(
  Kingdom = rep("Bacteria", 5),
  Genus = paste0("Genus", 1:5),
  row.names = paste0("ASV", 1:5)
  )

result <- process_ngs(
  X = mock_X,
  sample_data = mock_sample_data,
  taxa_table = mock_taxa_table,
  normalise = "load",
  load_colname = "load",
  transform = "none",
  verbose = FALSE
  )
}

}
\references{
#' McMurdie, P. J., & Holmes, S. (2013).
phyloseq: An R package for reproducible interactive analysis and graphics of microbiome census data.
\emph{PLoS ONE}, 8(4), e61217. \doi{10.1371/journal.pone.0061217}

Martín-Fernández, J. A., Hron, K., Templ, M., Filzmoser, P., & Palarea-Albaladejo, J. (2015).
Bayesian-multiplicative treatment of count zeros in compositional data sets.
\emph{Statistical Modelling}, 15(2), 134–158. \doi{10.1177/1471082X14535524}

Palarea-Albaladejo, J., & Martín-Fernández, J. A. (2015).
zCompositions—R package for multivariate imputation of left-censored data under a compositional approach.
\emph{Chemometrics and Intelligent Laboratory Systems}, 143, 85–96. \doi{10.1016/j.chemolab.2015.02.019}

Gloor, G. B., Macklaim, J. M., Pawlowsky-Glahn, V., & Egozcue, J. J. (2017).
Microbiome datasets are compositional: And this is not optional.
\emph{Frontiers in Microbiology}, 8, 2224. \doi{10.3389/fmicb.2017.02224}

Vandeputte, D., Kathagen, G., D’hoe, K., Vieira-Silva, S., Valles-Colomer, M., Sabino, J.,
Wang, J., Tito, R. Y., De Commer, L., Darzi, Y., Vermeire, S., Falony, G., & Raes, J. (2017).
Quantitative microbiome profiling links gut community variation to microbial load. \emph{Nature},
551(7681), 507–511. \doi{10.1038/nature24460}
}
\seealso{
\itemize{
\item \code{\link[=build_phyloseq]{build_phyloseq()}}
\item \code{\link[zCompositions:cmultRepl]{zCompositions::cmultRepl()}}
}
}
