% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ready_plots.R
\name{ready_plots}
\alias{ready_plots}
\title{Generate plots from a differential analysis (\code{dana}) object}
\usage{
ready_plots(
  dana_obj,
  term_name,
  pval_match,
  alpha = 0.1,
  add_interactions = TRUE,
  add_labels = TRUE,
  plot_coeff = TRUE,
  plot_feat = TRUE,
  plot_ranef = FALSE,
  X_colnames = NULL,
  sdata_var = NULL,
  group_colours = NULL,
  paired_id = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{dana_obj}{A \code{dana} object returned by \code{dana()}, containing model results.}

\item{term_name}{The name of the model term to plot (e.g., \code{"group"} or \code{"group:time"}).}

\item{pval_match}{Regex pattern to match the desired P value column in the results.}

\item{alpha}{Numeric. Significance threshold to consider features for plotting. Default 0.1.}

\item{add_interactions}{Logical. Whether to include interaction terms related to \code{term_name}.}

\item{add_labels}{Logical. Whether to add custom feature labels in plots.
A "feat_name" or "taxon_name" column must be in the \code{dana} object.
See \code{add_taxa()} and \code{add_feat_name()}.}

\item{plot_coeff}{Logical. Whether to generate coefficient-level plots.
Will generate volcano, heatmap and dot plots.}

\item{plot_feat}{Logical. Whether to generate feature-level plots for a specific
variable in \code{sample_data}.}

\item{plot_ranef}{Logical. Whether to generate random effect variance plots.
Only for mixed-effects models.}

\item{X_colnames}{Optional. Character vector specifying which features from \code{X} to plot.
If \code{NULL} and \code{plot_feat = TRUE} (the default), top 10 features based on P value are selected.}

\item{sdata_var}{Character. A column in \code{dana_obj$sdata} used for feature-level
plots when \code{plot_feat = TRUE}.}

\item{group_colours}{Optional named vector of colours for \code{sdata_var} groups
to be passed as \code{values} argument to \code{ggplot2::scale_fill_manual()}.}

\item{paired_id}{Optional. Column name in \code{sdata} specifying sample pairing (e.g., subject_id).}

\item{verbose}{Logical. Whether to display messages during processing.}

\item{...}{Additional \code{ggplot2::theme()} arguments passed to internal plotting helpers (e.g., font sizes).}
}
\value{
A named list of \code{ggplot} objects stored in \code{dana_obj$plots}. These may include:
\itemize{
\item \code{coeff_volcano}, \code{coeff_heatmap}, \code{coeff_point}
\item \code{feat_scatter}, \code{feat_boxplot}, \code{feat_violin}, \code{feat_ridge}
\item \code{ranef_all}
}
}
\description{
This function produces a range of coefficient- and feature-level plots from a \code{dana} object
for a given model term of interest. It supports both main effect and interaction terms,
and can visualize significant results from either \code{fit} or \code{lrt} P values.
}
\details{
When \code{add_interactions = TRUE}, the function shows \code{fit} coefficients that
match significant main and interaction terms.

If no significant features are found under the specified \code{alpha} significance
threshold, the function will abort.
}
\examples{
set.seed(123)
mock_X <- matrix(rnorm(20 * 5), nrow = 20)
colnames(mock_X) <- paste0("feat_", seq_len(5))
rownames(mock_X) <- paste0("sample_", seq_len(20))

sample_data <- data.frame(
  sample_id = rownames(mock_X),
  group = factor(rep(c("A", "B"), each = 10)),
  time = factor(rep(c("T1", "T2"), times = 10)),
  subject_id = factor(rep(seq_len(10), each = 2)),
  stringsAsFactors = FALSE
)
rownames(sample_data) <- sample_data$sample_id

fit_df <- data.frame(
  feat_id = rep(colnames(mock_X), each = 2),
  Coefficient = rep(c("(Intercept)", "groupB"), 5),
  Estimate = rnorm(10),
  `Pr(>|t|)` = runif(10),
  padj = runif(10),
  stringsAsFactors = FALSE
)

# Mock dana object
dana_obj <- list(
  X = mock_X,
  sdata = sample_data,
  formula_rhs = ~ group,
  fit = fit_df,
  lrt = data.frame(),  #' empty but valid
  ranef = data.frame() #' empty but valid
)
class(dana_obj) <- "dana"

dana_obj <- dana_obj |>
ready_plots(
  term_name = "group",
  pval_match = "padj",
  alpha = 0.5,
  add_labels = FALSE,
  plot_coeff = TRUE,
  plot_feat = TRUE,
  plot_ranef = FALSE,
  sdata_var = "group",
  verbose = FALSE
)

# Visualize generated plots
dana_obj$plots

}
\seealso{
\itemize{
\item \code{\link[=dana]{dana()}} for fitting differential analysis models on omics datasets.
\item \code{\link[=add_taxa]{add_taxa()}} and \code{\link[=add_feat_name]{add_feat_name()}} for adding feature labels to dana object.
\item \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}} and \code{\link[ggplot2:theme]{ggplot2::theme()}} to further customise plots.
}
}
