% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psymet_functions.R
\name{cronbachs}
\alias{cronbachs}
\title{Cronbach's Alpha for Multiple Sets of Variables/Items}
\usage{
cronbachs(
  data,
  vrb.nm.list,
  ci.type = "delta",
  level = 0.95,
  use = "pairwise.complete.obs",
  stats = c("average_r", "nvrb"),
  R = 200L,
  boot.ci.type = "perc"
)
}
\arguments{
\item{data}{data.frame of data.}

\item{vrb.nm.list}{list of character vectors specifying the sets of
variables/items. Each element of \code{vrb.nm.list} provides the colnames
of \code{data} for that set of variables/items.}

\item{ci.type}{character vector of length 1 specifying the type of confidence
interval to compute. The options are 1) "classic" = the Feldt et al. (1987)
procedure using only the mean covariance, 2) "delta" = the Duhhacheck &
Iacobucci (2004) procedure using the delta method of the covariance matrix,
or 3) "boot" = bootstrapped confidence intervals with the method specified
by \code{boot.ci.type}.}

\item{level}{double vector of length 1 with a value between 0 and 1
specifying what confidence level to use.}

\item{use}{character vector of length 1 specifying how to handle missing data
when computing the covariances. The options are 1) "pairwise.complete.obs",
2) "complete.obs", 3) "na.or.complete", 4) "all.obs", or 5) "everything".
See details of \code{\link{cov}}.}

\item{stats}{character vector specifying the additional statistical
information you could like related to cronbach's alpha. Options are: 1)
"std.alpha" = cronbach's alpha of the standardized variables/items, 2)
"G6(smc)" = Guttman's Lambda 6 reliability, 3) "average_r" = mean
correlation between the variables/items, 4) "median_r" = median correlation
between the variables/items, 5) "mean" = mean of the the scores from
averaging the variables/items together, 6) "sd" = standard deviation of the
scores from averaging the variables/items together, 7) "nvrb" = number of
variables/items. The default is "average_r" and "nvrb".}

\item{R}{integer vector of length 1 specifying the number of bootstrapped
resamples to do. Only used when \code{ci.type} = "boot".}

\item{boot.ci.type}{character vector of length 1 specifying the type of
bootstrapped confidence interval to compute. The options are 1) "perc" for
the regular percentile method, 2) "bca" for bias-corrected and accelerated
percentile method, 3) "norm" for the normal method that uses the
bootstrapped standard error to construct symmetrical confidence intervals
with the classic formula around the bias-corrected estimate, and 4) "basic"
for the basic method. Note, "stud" for the studentized method is NOT an
option. See \code{\link[boot]{boot.ci}} as well as
\code{\link{confint2.boot}} for details.}
}
\value{
data.frame containing the following columns:

\describe{
   \item{est}{Cronbach's alpha itself}
   \item{se}{standard error for Cronbach's alpha}
   \item{lwr}{lower bound of the confidence interval of Cronbach's alpha}
   \item{upr}{upper bound for the confidence interval of Cronbach's alpha},
   \item{???}{any statistics requested via the \code{stats} argument}
}
}
\description{
\code{cronbachs} computes Cronbach's alpha for multiple sets of
variables/items as an estimate of reliability for multiple scores. There are
three different options for confidence intervals. Missing data can be handled
by either pairwise deletion (\code{use} = "pairwise.complete.obs") or
listwise deletion (\code{use} = "complete.obs"). \code{cronbachs} is a
wrapper for the \code{\link[psych]{alpha}} function in the \code{psych}
package.
}
\details{
When \code{ci.type} = "classic" the confidence interval is based on the mean
covariance. It is the same as the confidence interval used by
\code{\link[psych]{alpha.ci}} (Feldt, Woodruff, & Salih, 1987). When
\code{ci.type} = "delta" the confidence interval is based on the delta method
of the covariance matrix. It is based on the standard error returned by
\code{\link[psych]{alpha}} (Duhachek & Iacobucci, 2004).
}
\examples{

dat0 <- psych::bfi
dat1 <- str2str::pick(x = dat0, val = c("A1","C4","C5","E1","E2","O2","O5",
   "gender","education","age"), not = TRUE, nm = TRUE)
vrb_nm_list <- lapply(X = str2str::sn(c("E","N","C","A","O")), FUN = function(nm) {
   str2str::pick(x = names(dat1), val = nm, pat = TRUE)})
cronbachs(data = dat1, vrb.nm.list = vrb_nm_list, ci.type = "classic")
cronbachs(data = dat1, vrb.nm.list = vrb_nm_list, ci.type = "delta")
cronbachs(data = dat1, vrb.nm.list = vrb_nm_list, ci.type = "boot")
suppressMessages(cronbachs(data = attitude, vrb.nm.list =
   list(names(attitude)))) # also works with only one set of variables/items

}
\references{
Feldt, L. S., Woodruff, D. J., & Salih, F. A. (1987). Statistical inference for
coefficient alpha. Applied Psychological Measurement (11) 93-103.

Duhachek, A. and Iacobucci, D. (2004). Alpha's standard error (ase): An accurate
and precise confidence interval estimate. Journal of Applied Psychology, 89(5):792-808.
}
\seealso{
\code{\link{cronbach}}
   \code{\link{composites}}
}
