#' @title
#' Computing transitive closure.
#'
#' @description
#' Computes the transitive closure of the input binary relation.
#'
#' @param rel A two-columns character matrix, each row comprising an element (pair) of the binary relation.
#'
#' @return
#' A transitive binary relation, as a two-columns character matrix (each row comprises an element (pair)
#' of the transitivity closed relation).
#'
#' @examples
#' rel <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "d", "a",
#'   "c", "a",
#'   "a", "a",
#'   "b", "b",
#'   "c", "c",
#'   "d", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' t.clo<-TransitiveClosure(rel)
#'
#' @name TransitiveClosure
#' @export TransitiveClosure
TransitiveClosure <- function(rel) {
  if (ncol(rel) != 2)
    stop("rel must be two columns character matrix")
  if ((ncol(rel) * nrow(rel) > 0  && !is.character(rel)))
    stop("rel must be two columns character matrix")

  tryCatch({
    result <- .Call("_TransitiveClosure", rel)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
