% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misclassification.R
\name{misclass}
\alias{misclass}
\alias{probsens}
\title{Misclassification of exposure or outcome}
\usage{
misclass(
  case,
  exposed,
  type = c("exposure", "exposure_pv", "outcome"),
  bias_parms = NULL,
  alpha = 0.05
)

probsens(
  case,
  exposed,
  type = c("exposure", "exposure_pv", "outcome"),
  reps = 1000,
  seca = list(dist = c("constant", "uniform", "triangular", "trapezoidal", "normal",
    "beta"), parms = NULL),
  seexp = NULL,
  spca = list(dist = c("constant", "uniform", "triangular", "trapezoidal", "normal",
    "beta"), parms = NULL),
  spexp = NULL,
  corr_se = NULL,
  corr_sp = NULL,
  alpha = 0.05
)
}
\arguments{
\item{case}{Outcome variable. If a variable, this variable is tabulated against.}

\item{exposed}{Exposure variable.}

\item{type}{Choice of misclassification:
\enumerate{
\item exposure: bias analysis for exposure misclassification; corrections
using sensitivity and specificity: nondifferential and independent errors,
\item exposure_pv: bias analysis for exposure misclassification; corrections
using PPV/NPV: nondifferential and independent errors,
\item outcome: bias analysis for outcome misclassification.
}}

\item{bias_parms}{Vector defining the bias parameters. This vector has 4
elements between 0 and 1, in the following order:
\enumerate{
\item Sensitivity of exposure (when \code{type = "exposure"}) or outcome
(when \code{type = "outcome"}) classification among those with the outcome
(when \code{type = "exposure"}) or exposure (when \code{type = "outcome"}),
\item Sensitivity of exposure (or outcome) classification among those without
the outcome (or exposure),
\item Specificity of exposure (or outcome) classification among those with the
outcome (or exposure), and
\item Specificity of exposure (or outcome) classification among those without
the outcome (or exposure).
}
If PPV/NPV is chosen in case of exposure misclassification, this vector is the
following:
\enumerate{
\item Positive predictive value among those with the outcome,
\item Positive predictive value among those without the outcome,
\item Negative predictive value among those with the outcome,
\item Negative predictive value among those without the outcome.
}}

\item{alpha}{Significance level.}

\item{reps}{Number of replications to run.}

\item{seca}{List defining sensitivity among cases:
\enumerate{
\item The sensitivity of exposure classification among those with the outcome
(when \code{type = "exposure"}), or
\item The sensitivity of outcome classification among those with the exposure
(when \code{type = "outcome"}).
}
The first argument provides the probability distribution function (constant,
uniform, triangular, trapezoidal, truncated normal, or beta) and the second
its parameters as a vector. Lower and upper bounds of the truncated normal
have to be between 0 and 1.
\enumerate{
\item constant: constant value,
\item uniform: min, max,
\item triangular: lower limit, upper limit, mode,
\item trapezoidal: min, lower mode, upper mode, max,
\item normal: lower bound, upper bound, mean, sd.
\item beta: alpha, beta.
}
If PPV/NPV is chosen in case of exposure misclassification, the same four (4)
parameters \code{seca}, \code{seexp}, \code{spca}, \code{spexp} as for Se/Sp have to be used but
with the following meaning, and only for a beta distributions and no
correlation between distributions:
\enumerate{
\item Positive predictive value among those with the outcome,
\item Positive predictive value among those without the outcome,
\item Negative predictive value among those with the outcome,
\item Negative predictive value among those without the outcome.
}}

\item{seexp}{List defining sensitivity among controls:
\enumerate{
\item The sensitivity of exposure classification among those without the
outcome (when \code{type = "exposure"}), or
\item The sensitivity of outcome classification among those without the
exposure (when \code{type = "outcome"}).
}}

\item{spca}{List as above for \code{seca} but for specificity.}

\item{spexp}{List as above for \code{seexp} but for specificity.}

\item{corr_se}{Correlation between case and non-case sensitivities. If PPV/NPV is
chosen in case of exposure misclassification, correlations are set to NULL.}

\item{corr_sp}{Correlation between case and non-case specificities.}
}
\value{
A list with elements (for \code{misclass()}):
\item{obs_data}{The analyzed 2 x 2 table from the observed data.}
\item{corr_data}{The expected observed data given the true data assuming misclassification.}
\item{obs_measures}{A table of observed relative risk and odds ratio with
confidence intervals.}
\item{adj_measures}{A table of corrected relative risks and odds ratios.}
\item{bias_parms}{Input bias parameters.}

A list with elements (for \code{probsens()}):
\item{obs_data}{The analyzed 2 x 2 table from the observed data.}
\item{obs_measures}{A table of observed relative risk and odds ratio with
confidence intervals.}
\item{adj_measures}{A table of corrected relative risks and odds ratios.}
\item{sim_df}{Data frame of random parameters and computed values.}
\item{reps}{Number of replications.}
}
\description{
\code{misclass()} and \code{probsens()} allow to provide adjusted measures
of association corrected for misclassification of the exposure or the outcome.
}
\section{Simple bias analysis with \code{misclass()}}{

\code{misclass()} allows you to run a simple sensitivity analysis for disease or
exposure misclassification. Confidence interval for odds ratio adjusted using
sensitivity and specificity is computed as in Chu et al. (2006), for exposure
misclassification.

For exposure misclassification, bias-adjusted measures are available using
sensitivity and specificity, or using predictive values.
}

\section{Probabilistic sensitivity analysis with \code{probsens()}}{

\code{probsens()} performs a summary-level probabilistic sensitivity analysis to
correct for exposure misclassification or outcome misclassification and random
error. Non-differential misclassification is assumed when only the two bias
parameters \code{seca} and \code{spca} are provided. Adding the 2 parameters
\code{seexp} and \code{spexp} (i.e. providing the 4 bias parameters) evaluates
a differential misclassification.

For exposure misclassification, bias-adjusted measures are available using
sensitivity and specificity, or using predictive values. However, only a beta
distribution is available for predictive values.

Correlations between sensitivity (or specificity) of exposure classification
among cases and controls can be specified and use the NORmal To Anything
(NORTA) transformation (Li & Hammond, 1975).

In case of negative (<=0) adjusted count in the 2-by-2 table following given
prior Se/Sp distribution(s), draws are discarded.
}

\section{Updated calculations, probabilistic bias analysis}{

episensr 2.0.0 introduced updated calculations of probabilistic bias analyses
by (1) using the NORTA transformation to define a correlation between
distributions, and (2) sampling true prevalences and then sampling the
adjusted cell counts rather than just using the expected cell counts from a
simple quantitative bias analysis. This updated version should be preferred
but if you need to run an old analysis, you can easily revert to the
computation using \code{\link[=probsens_legacy]{probsens_legacy()}} as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(episensr)
probsens <- probsens_legacy
}\if{html}{\out{</div>}}
}

\examples{
# The data for this example come from:
# Fink, A.K., Lash,  T.L. A null association between smoking during pregnancy
# and breast cancer using Massachusetts registry data (United States).
# Cancer Causes Control 2003;14:497-503.
misclass(matrix(c(215, 1449, 668, 4296),
dimnames = list(c("Breast cancer+", "Breast cancer-"),
c("Smoker+", "Smoker-")),
nrow = 2, byrow = TRUE),
type = "exposure",
bias_parms = c(.78, .78, .99, .99))

misclass(matrix(c(4558, 3428, 46305, 46085),
dimnames = list(c("AMI death+", "AMI death-"),
c("Male+", "Male-")),
nrow = 2, byrow = TRUE),
type = "outcome",
bias_parms = c(.53, .53, .99, .99))

# The following example comes from Chu et al. Sensitivity analysis of
# misclassification: A graphical and a Bayesian approach.
# Annals of Epidemiology 2006;16:834-841.
misclass(matrix(c(126, 92, 71, 224),
dimnames = list(c("Case", "Control"), c("Smoker +", "Smoker -")),
nrow = 2, byrow = TRUE),
type = "exposure",
bias_parms = c(.94, .94, .97, .97))

# The next example, using PPV/NPV, comes from Bodnar et al. Validity of birth
# certificate-derived maternal weight data.
# Paediatric and Perinatal Epidemiology 2014;28:203-212.
misclass(matrix(c(599, 4978, 31175, 391851),
dimnames = list(c("Preterm", "Term"), c("Underweight", "Normal weight")),
nrow = 2, byrow = TRUE),
type = "exposure_pv",
bias_parms = c(0.65, 0.74, 1, 0.98))
#
# The data for this example come from:
# Greenland S., Salvan A., Wegman D.H., Hallock M.F., Smith T.J.
# A case-control study of cancer mortality at a transformer-assembly facility.
# Int Arch Occup Environ Health 1994; 66(1):49-54.
greenland <- matrix(c(45, 94, 257, 945), dimnames = list(c("BC+", "BC-"),
c("Smoke+", "Smoke-")), nrow = 2, byrow = TRUE)
set.seed(123)
# Exposure misclassification, non-differential
probsens(greenland, type = "exposure", reps = 20000,
seca = list("trapezoidal", c(.75, .85, .95, 1)),
spca = list("trapezoidal", c(.75, .85, .95, 1)))

# Exposure misclassification, differential
probsens(greenland, type = "exposure", reps = 20000,
seca = list("trapezoidal", c(.75, .85, .95, 1)),
seexp = list("trapezoidal", c(.7, .8, .9, .95)),
spca = list("trapezoidal", c(.75, .85, .95, 1)),
spexp = list("trapezoidal", c(.7, .8, .9, .95)),
corr_se = .8,
corr_sp = .8)

probsens(greenland, type = "exposure", reps = 20000,
seca = list("beta", c(908, 16)),
seexp = list("beta", c(156, 56)),
spca = list("beta", c(153, 6)),
spexp = list("beta", c(205, 18)),
corr_se = .8,
corr_sp = .8)

probsens(matrix(c(338, 490, 17984, 32024),
dimnames = list(c("BC+", "BC-"), c("Smoke+", "Smoke-")), nrow = 2, byrow = TRUE),
type = "exposure",
reps = 1000,
seca = list("trapezoidal", c(.8, .9, .9, 1)),
spca = list("trapezoidal", c(.8, .9, .9, 1)))

# Disease misclassification
probsens(matrix(c(173, 602, 134, 663),
dimnames = list(c("BC+", "BC-"), c("Smoke+", "Smoke-")), nrow = 2, byrow = TRUE),
type = "outcome",
reps = 20000,
seca = list("uniform", c(.8, 1)),
spca = list("uniform", c(.8, 1)))

probsens(matrix(c(338, 490, 17984, 32024),
dimnames = list(c("BC+", "BC-"), c("Smoke+", "Smoke-")), nrow = 2, byrow = TRUE),
type = "outcome",
reps = 20000,
seca = list("uniform", c(.2, .6)),
seexp = list("uniform", c(.1, .5)),
spca = list("uniform", c(.99, 1)),
spexp = list("uniform", c(.99, 1)),
corr_se = .8,
corr_sp = .8)

probsens(matrix(c(173, 602, 134, 663),
dimnames = list(c("BC+", "BC-"), c("Smoke+", "Smoke-")), nrow = 2, byrow = TRUE),
type = "outcome",
reps = 20000,
seca = list("beta", c(100, 5)),
seexp = list("beta", c(110, 10)),
spca = list("beta", c(120, 15)),
spexp = list("beta", c(130, 30)),
corr_se = .8,
corr_sp = .8)

# Fox M.P., MacLehose R.F., Lash T.L.
# SAS and R code for probabilistic quantitative bias analysis for
# misclassified binary variables and binary unmeasured confounders
# Int J Epidemiol 2023:1624-1633.
\dontrun{
fox <- matrix(c(40, 20, 60, 80),
dimnames = list(c("Diseased", "Non-diseased"), c("Exposed", "Unexposed")),
nrow = 2, byrow = TRUE)
set.seed(1234)
probsens(fox, type = "exposure", reps = 10^6,
seca = list("beta", c(25, 3)),
spca = list("trapezoidal", c(.9, .93, .97, 1)),
seexp = list("beta", c(47, 7)),
spexp = list("trapezoidal", c(.8, .83, .87, .9)),
corr_se = .8,
corr_sp = .8)
}

# Using PPV/NPV, from Bodnar et al. Validity of birth certificate-derived maternal
# weight data. Paediatric and Perinatal Epidemiology 2014;28:203-212.
set.seed(1234)
probsens(matrix(c(599, 4978, 31175, 391851),
dimnames = list(c("Preterm", "Term"), c("Underweight", "Normal weight")),
nrow = 2, byrow = TRUE),
type = "exposure_pv", reps = 10^6,
seca = list("beta", c(50, 27)),  ## PPV_case
spca = list("beta", c(120, .5)),  ## NPV_case
seexp = list("beta", c(132, 47)),  ## PPV_ctrl
spexp = list("beta", c(115, 2)))  ## NPV_ctrl
}
\references{
Fox, M.P, MacLehose, R.F., Lash, T.L., 2021 \emph{Applying Quantitative
Bias Analysis to Epidemiologic Data}, pp.141--176, 233--256, 293--308, Springer.

Li, S.T., Hammond, J.L., 1975. \emph{Generation of Pseudorandom Numbers
with Specified Univariate Distributions and Correlation Coefficients}.
IEEE Trans Syst Man Cybern 5:557-561.

Chu, H., Zhaojie, W., Cole, S.R., Greenland, S., \emph{Sensitivity analysis of
misclassification: A graphical and a Bayesian approach},
Annals of Epidemiology 2006;16:834-841.

Barros, A. & Hirakata, V.N., 2003. Alternatives for Logistic Regression in
Cross-sectional Studies: An Empirical Comparison of Models that Directly
Estimate the Prevalence Ratio. BMC Medical Research Methodology 3:21.

McNutt, L-A, Wu, C., Xue, X., Hafner J.P., 2003. \emph{Estimating the Relative
Risk in Cohort Studies and Clinical Trials of Common Outcomes}. American
Journal of Epidemiology 157(10):940-943.

Greenland, S. (2004). Model-based Estimation of Relative Risks and Other
Epidemiologic Measures in Studies of Common Outcomes and in Case-Control
Studies. American Journal of Epidemiology 160(4):301-305.

Zhou, G. (2004). A Modified Poisson Regression Approach to Prospective Studies
with Binary Data. American Journal of Epidemiology 159(7):702-706.
}
\seealso{
Other misclassification: 
\code{\link{misclass_cov}()},
\code{\link{probsens.irr}()}
}
\concept{misclassification}
