#' Details on Broadcasted Operators
#' 
#' @description
#' 
#' Base 'R' comes with relational (==, !=, etc.),
#' arithmetic (+, -, *, /, etc.), and logical/bit-wise (&, |) operators. \cr
#' 'broadcast' provides 2 ways to use these operators with broadcasting. \cr
#' \cr
#' The first (and simple) way is to use the \link{broadcaster} class,
#' which comes with it's own method dispatch for the above mentioned operators. \cr
#' This approach supports operator precedence, and for the average 'R' user,
#' this is sufficient. \cr
#' \cr
#' The second way is to use the large set of `bc.`- functions. \cr
#' These offer much greater control and more operators than the previous method,
#' and has less risk of running into conflicting methods. \cr
#' But it does not support operator precedence. \cr
#' \cr
#' 
#' 
#' @section Operators Overloaded via Broadcaster Class: 
#' The 'broadcast' package provides the \link{broadcaster} class,
#' which comes with its own method dispatch for the base operators. \cr
#' If at least one of the 2 arguments of the base operators has the \link{broadcaster} class attribute,
#' and no other class (like `bit64`) interferes,
#' broadcasting will be used. \cr
#' \cr
#' The following arithmetic operators have a 'broadcaster' method:
#' +, -, *, /, ^, %%, %/% \cr
#' The following relational operators have a 'broadcaster' method:
#' `r paste0(broadcast:::.op_rel(), collapse = ", ")` \cr
#' And finally, the & and | operators also have a 'broadcaster' method. \cr
#' \cr
#' As the \link{broadcaster} operator methods simply overload the base operators,
#' operator precedence rules are preserved for the \link{broadcaster} operator methods. \cr
#' \cr
#' See also the Examples section below. \cr
#' \cr
#' 
#' 
#' @section Available `bc.` functions: 
#' 'broadcast' provides a set of functions for broadcasted element-wise binary operations
#' with broadcasting. \cr
#' These functions use an API similar to the \link[base]{outer} function. \cr
#' \cr
#' The following functions for simple operations are available:
#' 
#'  * \link{bc.rel}: General relational operations.
#'  * \link{bc.b}: Boolean (i.e. logical) operations;
#'  * \link{bc.i}: integer arithmetic operations;
#'  * \link{bc.d}: decimal arithmetic operations;
#'  * \link{bc.cplx}: complex arithmetic operations;
#'  * \link{bc.str}: string (in)equality, concatenation, and distance operations;
#'  * \link{bc.raw}: operations that take in arrays of type `raw` and return an array of type `raw`;
#'  * \link{bc.bit}: BIT-WISE operations, supporting the `raw` and `integer` types;
#'  * \link{bc.list}: apply any 'R' function to 2 recursive arrays with broadcasting. \cr \cr
#' 
#' 
#' Note that the \link{bc.rel} method is the primary method for relational operators
#' (`r paste0(broadcast:::.op_rel(), collapse = ", ")`),
#' and provides what most user usually need in relational operators.\cr
#' The various other `bc.` methods have specialized relational operators available for very specialised needs. \cr
#' \cr
#' 
#' 
#' 
#' @section Attribute Handling: 
#' The `bc.` functions and the overloaded operators generally do \bold{not}
#' preserve attributes, unlike the base 'R' operators,
#' except for (dim)names and the \link{broadcaster} class attribute (and related attributes). \cr
#' \cr
#' Broadcasting often results in an object with more dimensions, larger dimensions,
#' and/or larger length than the original objects. \cr
#' This is relevant as some class-specific attributes are only appropriate for certain dimensions or lengths. \cr
#' Custom matrix classes, for example, presumes an object to have exactly 2 dimensions. \cr
#' And the various classes provided by the 'bit' package have length-related attributes. \cr
#' So class attributes cannot be guaranteed to hold for the resulting objects when broadcasting is involved. \cr
#' \cr
#' The `bc.` functions and the overloaded operators
#' \bold{always} preserve the "broadcaster" attribute,
#' as this is necessary to chain together broadcasted operations. \cr
#' \cr
#' Almost all functions provided by 'broadcast' are S3 or S4 generics; \cr
#' methods can be written for them for specific classes,
#' so that class-specific attributes can be supported when needed. \cr
#' \cr
#' Unary operations (i.e. `+ x`, `- x`) return the original object,
#' with only the sign adjusted. \cr
#' \cr
#' \cr
#' 
#' @example inst/examples/broadcaster.R
#' 
#' 
#' @name aaa01_broadcast_operators
#' @rdname aaa01_broadcast_operators
#' @aliases broadcast_operators
#' 
NULL
#> NULL
